;; Copyright © 2021 Atomist, Inc.
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns atomist.dockerhub
  (:require
   [goog.string :as gstring]
   [http.client :as client]
   [atomist.time]
   [cljs.pprint :refer [pprint]]
   [atomist.cljs-log :as log]
   [atomist.async :refer-macros [go-safe <?]]
   [clojure.string :as str]
   [atomist.docker :as docker]
   [atomist.time :as atm-time]
   [atomist.api :as api]
   [cljs-time.core :as cljs-time]
   [atomist.json :as json]))

(enable-console-print!)

(def auth-url "https://auth.docker.io/token")
(def domain "registry-1.docker.io")
(def timeout 5000)

(defn dockerhub-auth
  [{:keys [repository username api-key]}]
  (go-safe
   (when (and username api-key)
     (log/infof "Authenticating using username: %s" username)
     (let [repository (if (str/includes? repository "/") repository (str "library/" repository))
           response (<? (client/post auth-url
                                     {:timeout timeout
                                      :form-params {:service "registry.docker.io"
                                                    :client_id "Atomist"
                                                    :grant_type "password"
                                                    :username username
                                                    :password api-key
                                                    :scope (gstring/format "repository:%s:pull" repository)}}))]

       (if (= 200 (:status response))
         (do
           (log/infof "Logged in to Dockerhub as %s - found access-token? " username (string? (-> response :body :access_token)))
           {:access-token (-> response :body :access_token)
            :repository repository})
         (do
           (log/warnf "Unable to auth with dockerhub as %s: %s" username (pr-str response))
           (throw (ex-info (gstring/format "unable to auth dockerhub at %s" auth-url) response))))))))

(defn dockerhub-anonymous-auth
  [{:keys [repository username api-key]}]
  (go-safe
   (when (or
          (not username)
          (not api-key))
     (log/infof "Attempting anonymous auth for %s" repository)
     (let [repository (if (str/includes? repository "/") repository (str "library/" repository))
           response (<? (client/get (gstring/format "https://auth.docker.io/token?service=registry.docker.io&scope=repository:%s:pull" repository) {:timeout timeout}))]
       (if (= 200 (:status response))
         {:access-token (-> response :body :token)
          :repository repository}
         (throw (ex-info (gstring/format "unable to auth %s" auth-url) response)))))))

(defn private-repo?
  "Is repository a private repo?"
  [username api-key repository tag-or-digest]
  (go-safe
   (boolean
    (when (str/includes? repository "/")
      (if-let [{token :access-token} (<? (dockerhub-auth {:repository repository :username username :api-key api-key}))]
        (let [url (gstring/format "https://%s/v2/%s/manifests/%s" domain repository tag-or-digest)
              request-opts {:timeout timeout :headers {"Accept" "application/vnd.docker.distribution.manifest.v2+json,application/vnd.docker.distribution.manifest.list.v2+json,application/vnd.oci.image.manifest.v1+json,application/vnd.oci.image.index.v1+json"
                                                       "Authorization" (str "Bearer " token)}
                            :throw-exceptions false}
              response (<? (client/head url request-opts))]
          (if (or
               (some-> response :headers :www-authenticate (str/includes? "insufficient_scope"))
               (and (= 401 (:status response))
                    (some-> response :headers :www-authenticate (str/includes? "Bearer"))))
            (do
              (log/infof "Repository %s is private: %s" repository response)
              true)
            (do
              (log/infof "Repository %s probably public: %s" repository response)
              false)))
        (do
          (log/errorf "Error authenticating with Dockerhub with user %s" username)
          false))))))

(defn run-tasks [m & ts]
  (go-safe
   (loop [context m tasks ts]
     (if-let [task (first tasks)]
       (recur (merge context (<? (task context))) (rest tasks))
       context))))

(defn get-labelled-manifests
  "log error or return labels"
  ([repository tag-or-digest]
   (get-labelled-manifests repository tag-or-digest nil nil))
  ([repository tag-or-digest username api-key]
   (log/infof "get-image-info:  %s@%s/%s" (or username "anonymous") repository tag-or-digest)
   (go-safe
    (let [auth-context (<? (run-tasks
                            {:repository repository
                             :tag tag-or-digest
                             :username username
                             :api-key api-key}
                            dockerhub-auth
                            dockerhub-anonymous-auth))]
      (<? (docker/get-labelled-manifests domain (:access-token auth-context) (or (:repository auth-context) repository) tag-or-digest))))))

(defn get-repo-details
  "Get stuff stars, pulls etc"
  [repository]
  (go-safe
   (let [response (<? (client/get (gstring/format "https://hub.docker.com/v2/repositories/%s/" repository)))]
     (if (= 200 (:status response))
       (:body response)
       (log/warnf "Could not retrieve details for %s: %s" repository response)))))

(defn user-login
  "User old fashioned auth"
  [username api-key]
  (go-safe
   (log/infof "Logging in using username %s" username)
   (let [response (<? (client/post "https://hub.docker.com/v2/users/login"
                                   {:form-params {:username username
                                                  :password api-key}}))]
     (if (= 200 (:status response))
       (do
         (log/infof "Login successful")
         (-> response :body :token))
       (log/warnf "Unable to auth with dockerhub %s" response)))))

(defn last-tag
  [nspace repository-name access-token]
  (go-safe
   (log/infof "fetch tags for %s/%s" nspace repository-name)
   (->>
    (<? (client/get
         (gstring/format "https://hub.docker.com/v2/repositories/%s/%s/tags" nspace repository-name)
         {:headers
          {"Authorization" (str "Bearer " access-token)}}))
    :body
    :results
    (map #(assoc % :instant (atm-time/parse-weird-docker-datetime-string (:last_updated %))))
    (sort-by :instant cljs-time/after?)
    (first))))

(defn docker-days-ago [s]
  (try
    (->> s
         (atm-time/parse-weird-docker-datetime-string)
         (atm-time/days-ago))
    (catch :default _ nil)))

(defn decorate-repos
  [repos]
  (->> repos
       (map #(assoc % :days-since (-> % :last_updated docker-days-ago)))
       (filter #(not (nil? (:days-since %))))
       (sort-by :days-since)
       (into [])))

(defn ingest-latest-tags
  "config-change.edn: scan repositories and tags and ingest the most recent ones
   tag-discovery.edn: scan repositories on a schedule - skip transactions if latest tag has not been updated
     - page through repositories (api-usage n/10)
     - get latest tag (api-usage 1 per repo)
     - get manifests (api-usage 1 tag if image n / platform if manifest list)
     100 repositories where all of the latest tags are images (not manifest lists) -> 300 calls"
  [handler]
  (fn [{:as request :keys [docker-id namespace] :atomist/keys [docker-hub-creds
                                                               latest-tag-map]}]
    (when latest-tag-map
      (log/info "latest tag map:  " latest-tag-map))
    (go-safe
     (let [access-token (<? (user-login docker-id docker-hub-creds))
           repositories (->> (<? (docker/get-with-paging
                                  (gstring/format "https://hub.docker.com/v2/repositories/%s/" namespace)
                                  access-token
                                  10
                                  (fn [results body]
                                    (concat (or results []) (:results body)))))
                             decorate-repos)]
       (log/infof "Ingesting latest tag from %s repositories" (count repositories))
       (doseq [{repository :name} repositories
               :let [{tag :name
                      [{:keys [digest last_pushed]}] :images} (<? (last-tag namespace repository access-token))
                     repo-name (gstring/format "%s/%s" namespace repository)
                     repository-key ["hub.docker.com" repo-name]
                     last-pushed-js-date (-> last_pushed
                                             (atm-time/parse-weird-docker-datetime-string)
                                             (atm-time/->js-date))]]

         (log/infof "check %s last_pushed %s against %s" repository-key last-pushed-js-date (-> latest-tag-map (get repository-key)))
         (log/infof "check %s" (try (atm-time/<= last-pushed-js-date (-> latest-tag-map (get repository-key)))
                                    (catch :default ex
                                      (log/info "threw " ex)
                                      false)))
         (if (and latest-tag-map
                  (-> latest-tag-map (get repository-key))
                  (try
                    (atm-time/<= last-pushed-js-date (-> latest-tag-map (get repository-key)))
                    (catch :default _ false)))
           (log/infof "skipping tag (%s:%s@%s) because it's already transacted" repository tag digest)
           (do
             (log/infof "fetch tag (%s:%s)" tag digest)
             (when-let [manifests (not-empty
                                   (<? (docker/get-labelled-manifests domain access-token repo-name tag)))]
               (log/infof "Found %s manifests for %s:%s" (count manifests) repo-name digest)
               (doseq [manifest manifests
                       :let [new-digest (:digest manifest)]]
                 (log/infof "Digest for tag %s:%s platform %s -> %s" repo-name tag (:platform manifest) new-digest)
                 (<? (api/transact request (docker/->image-layers-entities-tagged
                                            "hub.docker.com"
                                            repo-name manifest tag
                                            last-pushed-js-date)))))))))
     (<? (handler (assoc request :atomist/status
                         {:code 0
                          :reason "Ingested latest tags"}))))))
