(ns atomist.api.v2.start
  (:require ["express" :as express]
            ["body-parser" :as body-parser]
            [clojure.edn :as edn]
            [atomist.time :as time] 
            [promesa.core :as p]
            [goog.string.format]
            [goog.string :refer [format]]
            [atomist.api.v2 :as v2]
            [atomist.promise :refer [chan->promise]]
            [atomist.api.v2.log :as log]
            [cljs.core.async :refer [<!] :refer-macros [go]]))

(defn ^:export start-http-listener
  "start a cloud-run handler
     params
       handler - (param) => Promise"
  [handler]
  (let [app (express)]
    (.use app (.text body-parser #js {:limit "1024mb" :type "application/edn"}))
    (.post app "/"
           (fn [req resp]
             ;; event-id for gcf
             ;; trace-id for CloudRun
             (let [start (time/now)
                   payload (edn/read-string (.. req -body))]
               (log/set-trigger payload)
               (p/catch
                (p/then
                 (handler payload)
                 (fn [r]
                   (p/then
                    (chan->promise
                     (v2/execution-patch
                      payload
                      {:status (or (:atomist/status r) {:state :failed :reason "missing status"})}))
                    (fn [_]
                      (.debug js/console (format "execution took %d ms, finished with status: 'ok'" (- (time/now) start)))
                      (.sendStatus ^js resp 201)))))
                (fn [err]
                  (p/catch
                   (p/then
                    (chan->promise
                     (v2/execution-patch
                      payload
                      {:status {:state :failed :reason (str "handler failure " err)}}))
                    (fn [_]
                      (.sendStatus ^js resp 201)))
                   (fn [_]
                     (println "failed to post handler failure")
                     (.sendStatus ^js resp 201))))))))
    (let [port (or (.. js/process -env -PORT) 8080)] 
      (.listen app port (fn [] (println "listening on port" port))))))
