(ns atomist.git
  (:require [cljs.core.async :refer [<! >! timeout chan close!]]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs-node-io.proc :as proc]
            [clojure.string :as s]
            ["os" :as os]
            ["tmp" :as tmp])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn from-git [context args]
  (go
    #_(log/debug "run " (->> args
                             (map #(-> %
                                       (s/replace #"https://(.*:x-oauth-basic)@(.*)" "$2")
                                       (s/replace #"https://(atomist:.*)@(.*)" "$2")))
                             (into [])))
    (if (not (empty? args))
      (let [ported-child-proc (proc/aexec (str "git " (apply str (interpose " " args))) {:cwd (:path context)})
            [err stdout stderr] (<! ported-child-proc)]
        (if err
          (merge context
                 {:error err
                  :exit-code (. err -code)
                  :stderr stderr
                  :stdout stdout})
          (merge context {:stdout stdout
                          :stderr stderr})))
      (merge context {:stdout "" :stderr ""}))))

(defn clone-ref [{:keys [path branch token owner repo sha]}]
  ;; need success
  (let [cmd ["clone"
             (gstring/format "https://%s@github.com/%s/%s.git"
                             (if (s/starts-with? token "v1.")
                               (str "atomist:" token)
                               (str token ":x-oauth-basic"))
                             owner repo)
             path
             "--depth" "1"]]
    (if (and branch (not sha))
      (concat cmd ["--branch" branch "--single-branch"])
      cmd)))

(defn sha [_]
  ;; need stdout
  ["rev-list" "-1" "HEAD" "--"])

(defn config [_ n v]
  ;; need success
  ["config" n v])

(defn add-all [_]
  ["add" "."])

(defn commit [_ message]
  ;; need success
  ["commit" "-m" (gstring/format "'%s'" message)])

(defn push [{:keys [branch]}]
  ["push" "--set-upstream" "origin" branch])

(defn has-branch [_ branch]
  ;; need branch to be in stdout
  ["branch" "--list" branch])

(defn create-branch [_ branch]
  ;; need both to be successful
  ["branch" branch])

(defn checkout [_ branch-or-sha]
  ["checkout" branch-or-sha "--"])

(defn ls-remote [_]
  ["ls-remote" "--heads"])

(defn fetch-branch [_ branch]
  ;; for refspecs, keep local and remote branches the same
  ["fetch" "origin" (gstring/format "'%s:%s'" branch branch) "--depth" "1"])

(defn fetch-sha [_ sha]
  ["fetch" "origin" sha])

(defn checkout-branch [_ branch]
  ["checkout" (gstring/format "'%s'" branch)])

(defn checkout-force [_ branch]
  ["checkout" "--force" (gstring/format "'%s'" branch)])

(defn porcelain-status [_]
  ["status" "--porcelain"])

(defn fetch-tags [_]
  ["fetch" "origin" "--depth" "1" "+refs/tags/*:refs/tags/*"])

(defn tag [{:keys [tag]}]
  ["tag" "-a" (gstring/format "--message=\"%s\"" tag) tag])

(defn push-tag [{:keys [tag]}]
  ["push" "origin" tag])

(defn git-describe-tags [{:keys [stdout]}]
  ["describe" "--tags" (s/trim stdout)])

(defn git-show-ref-tags [_]
  ["show-ref" "--tags"])

(defn git-rev-list [_]
  ["rev-list" "--tags" "--max-count=1"])

(defn git-ls-file [_ path]
  "output e.g: 100644 8eb534672302de55cf09662036f13fc092b3172d 0\tLICENSE"
  ["ls-files" "-s" path])