(ns atomist.cljs-log
  (:require [goog.string :as gstring]
            [goog.string.format]
            [cljs.pprint]
            [atomist.async :refer [<? go-safe]]
            [cljs.core.async :refer [>!] :as async :refer-macros [go]]
            ["@atomist/skill-logging/lib/logging" :as skill-logging]))

(def logger js/console)
(def atomist-logger)

(defn create-logger [config]
  (let [skill-logger ((. skill-logging -createLogger) (clj->js config))]
    (set! atomist-logger skill-logger)))

(defn from-promise
  "turn a promise into a channel
    - channel will emit Promise value or error and then close"
  [promise]
  (let [c (async/chan)]
    (.catch
     (.then promise (fn [result]
                      (go (>! c (or result {}))
                          (async/close! c))))
     (fn [error]
       (go (>! c (or error {}))
           async/close! c)))
    c))

(defn close-logger []
  (go-safe
   (<? (from-promise
        (if atomist-logger
          (.close atomist-logger)
          (js/Promise. (fn [accept _] (accept))))))))

(defn- ->log [k & args]
  ;; Send everything to user logs for now
  (let [s (apply str args)]
    (if atomist-logger
      (case k
        (:debug :trace) (.debug atomist-logger s)
        :info (.info atomist-logger s)
        :warning (.warn atomist-logger s)
        :error (.error atomist-logger s))
    ;; local logger for dev
      (case k
        :info (.info logger s)
        :error (.error logger s)
        :warning (.warn logger s)
        :debug (.debug logger s)
        :trace (.trace logger s)
        (.info logger s))))
  nil)

(defn- log [& args]
  (apply ->log :info args))

(defn warn [& args]
  (apply ->log :warning args))

(def info log)

(defn error [& args]
  (apply ->log :error args))

(defn debug [& args]
  (apply ->log :debug args))

(defn trace [& args]
  (apply ->log :trace args))

(defn infof [s & args]
  (info (apply gstring/format s (map #(or % nil) args))))

(defn warnf [s & args]
  (warn (apply gstring/format s (map #(or % nil) args))))

(defn debugf [s & args]
  (debug (apply gstring/format s (map #(or % nil) args))))

(defn errorf [ex s & args]
  (error (.-message ex))
  (error (.-stack ex))
  (error (apply gstring/format s (map #(or % nil) args))))

(defn ptrace [x]
  (trace (with-out-str (cljs.pprint/pprint x)))
  x)

(comment
  (str (.-stack (ex-info "what" {})))
  (create-logger {:correlationId "correlation-id" :eventId "event-id" :workspaceId "workspace-id" :skillId "skill-id"})
  (infof "hey %s" "Jude"))
