(ns atomist.gcp
  (:require [atomist.async :refer-macros [go-safe <?]]
            [cljs.core.async :as async]
            [atomist.cljs-log :as log]
            [http.client :as client]
            [goog.string :as gstring]
            [goog.string.format]
            [goog.crypt.base64 :as base64]
            ["jsonwebtoken" :as jwt]
            [atomist.time]
            [atomist.json :as json]
            [cljs-node-io.core :as io]
            [atomist.proc]
            [clojure.string]))

(defn creds->access-token
  "Get a google OAuth token from credentials.json file 
    params
     google-service-account must be json parsed and in clj form with keywords"
  [google-service-account]
  (go-safe
   (try
     (let [t (long (/ (atomist.time/now) 1000))
           data (clj->js {:iss (:client_email google-service-account)
                          :sub (:client_email google-service-account)
                          :aud "https://www.googleapis.com/oauth2/v4/token"
                          :iat t
                          :exp (+ t 60)
                          :scope "https://www.googleapis.com/auth/cloud-platform"})
           signed-jwt-token (.sign jwt
                                   data
                                   (:private_key google-service-account)
                                   #js {:algorithm "RS256"
                                        :keyid (:private_key_id google-service-account)})
           response (<? (client/post
                         "https://www.googleapis.com/oauth2/v4/token"
                         {:form-params {:grant_type "urn:ietf:params:oauth:grant-type:jwt-bearer"
                                        :assertion signed-jwt-token}
                          :throw-exceptions false}))]
       (if (= 200 (:status response))
         (do
           (log/infof "Successfully authenticated to google...")
           (-> response :body :access_token))
         (throw (ex-info (gstring/format "failure to authorize:  %s - %s" (:status response) (:body response)) {}))))
     (catch :default e
       (log/errorf e "Error getting access token")
       (throw (ex-info "Error getting access token" {}))))))

(defn workload-identity
  "no need to specify the `scopes` query param because the default
    https://www.googleapis.com/auth/cloud-platform is sufficient"
  []
  (go-safe
   (log/info "workload-identity")
   (let [response (<? (client/get
                       "http://metadata.google.internal/computeMetadata/v1/instance/service-accounts/default/token"
                       {:headers {"Metadata-Flavor" "Google"}}))]
     (if (= 200 (:status response))
       (-> response :body :access_token)
       (throw (ex-info "workoad-identity failed to retrieve default token" {:response response}))))))

(defn get-secret [secret-name access-token]
  (go-safe
   (log/info "get-secret")
   (let [response
         (<? (client/get
              (gstring/format 
                "https://secretmanager.googleapis.com/v1/projects/%s/secrets/%s/versions/latest:access" 
                "atomist-skill-production"
                secret-name)
              {:headers {"Authorization" (gstring/format "Bearer %s" access-token)}}))]
     (if (= 200 (:status response))
       (-> response :body :payload :data (base64/decodeString))
       (throw (ex-info "get-secret failed to retrieve secret" {:response response}))))))

(comment
  (async/go
    (let [[_ stdout stderr] (async/<! (atomist.proc/aexecFile "gcloud" ["auth" "print-access-token"] {}))]
      (def user-access-token (clojure.string/trim stdout))))
  ;; on gcf using workload-identity to get the access-token - test with user creds json file
  (async/go 
    (def secret (async/<! (get-secret "aws-assume-role-creds" user-access-token)))))
