(ns atomist.proc
  (:require [atomist.cljs-log :as log]
            [cljs.core.async :refer [chan >! <!] :refer-macros [go]]
            [clojure.string :as s]
            [cljs-node-io.core :as io]
            [cljs-node-io.proc :as proc]
            [goog.string.format]
            [goog.string :as gstring]))

(def childproc (js/require "child_process"))

(set! *warn-on-infer* false)

(defn- child-process [s ->sub-process]
  (let [c (chan)
        subprocess (->sub-process (fn [err stdout stderr]
                                    (go (>! c [err stdout stderr]))))]
    (.on (. subprocess -stdout) "data" (fn [d] (print (s/trim-newline (gstring/format "(%s):  %s" s (str d))))))
    (.on (. subprocess -stderr) "data" (fn [d] (print (s/trim-newline (gstring/format "(%s):  %s" s (str d))))))
    c))

(defn aexec [cmdline opts]
  (child-process
   (-> cmdline (s/split #" ") first)
   (fn [callback] (childproc.exec cmdline (clj->js opts) callback))))

(defn aexecFile [pathstr args opts]
  (child-process
   (-> pathstr (io/file) (.getName))
   (fn [callback] (childproc.execFile pathstr (into-array args) (clj->js opts) callback))))

(defn spawn [command args]
  (go
    (log/info "args " args)
    (let [c (chan)
          p (proc/spawn command args {})]
      (.on (.-stderr p) "data" (fn [d] (log/error d)))
      (.on (.-stdout p) "data" (fn [d] (log/info d)))
      (.on p "close" (fn [code]
                       (log/infof "%s stopped with code %s" command code)
                       (go
                         (if (= 0 code)
                           (>! c code)
                           (>! c (ex-info
                                  (gstring/format "%s failed (%s)" command code)
                                  {:code code
                                   :command command
                                   :args args}))))))
      (<! c))))

(comment
  (go
    (let [[_ stdout stderr] (<! (aexecFile "./atomist.sh" [] {:cwd "/Users/slim/atmhq/bruce"}))]
      (log/infof "finished %d %d" (count stdout) (count stderr)))))
