(ns atomist.github.database
  (:require [http.client :as client]
            [goog.string :as gstring]
            [goog.string.format]
            [clojure.string :as s]
            [atomist.cljs-log :as log]
            [atomist.json :as json]
            [goog.crypt.base64 :as b64]
            [cljs.spec.alpha :as spec]
            [atomist.async :refer-macros [go-safe <?]]
            [cljs.pprint :refer [pprint]]
            [cljs.core.async :refer [<!] :as async :refer-macros [go]]))

(defn- api-v3
  [{:keys [token ref check-status response body url method]}]
  (go-safe
    (let [u (url ref)
          r (<? (method u
                        (merge
                         {:headers {"Authorization" (gstring/format "bearer %s" token)
                                    "User-Agent" "atomist"
                                    "Accept" "application/vnd.github.v3+json"}}
                         (if body
                           {:body (json/->str body)}))))]
      (if (check-status (-> r :status))
        (if response
          (response r)
          (:body r))
        (do
          (log/warnf "Status %s:  %s -- %s" (:status r) u (:body r))
          (throw (ex-info (gstring/format "%s failed with status %s" u (:status r)) {:resopnse r}))))))) 

(defn new-commit
  [{:keys [ref] :as request :atomist/keys [tree message]}]
  (log/info "create commit for " tree " with message '" message "' and parent " (:sha ref))
  (api-v3 (assoc request
                 :url #(gstring/format "https://api.github.com/repos/%s/%s/git/commits"
                                       (:owner %)
                                       (:repo %))
                 :method client/post
                 :check-status #(= 201 %)
                 :response #(do
                              (assoc request :atomist/commit-sha (-> % :body :sha)))
                 :body {:tree (:sha tree)
                        :message (or message "Atomist commit")
                        :parents [(:sha ref)]})) )

(defn update-ref
  "not forcing if ref does not currently point at commit parent"
  [{:atomist/keys [commit-sha] :as request}]
  (log/info "try to move " commit-sha " to " (:ref request))
  (api-v3 (assoc request
                 :method client/patch
                 :url #(gstring/format "https://api.github.com/repos/%s/%s/git/refs/heads/%s"
                                       (:owner %)
                                       (:repo %)
                                       (:branch %))
                 :body {:sha commit-sha}
                 :check-status #(= 200 %))))

(defn- new-blob
  "construct a new blob - response body will have a sha for the content"
  [request content]
  (api-v3 (assoc request
                 :body {:content content}
                 :method client/post
                 :url #(gstring/format "https://api.github.com/repos/%s/%s/git/blobs"
                                       (:owner %)
                                       (:repo %))
                 :check-status (constantly true))))

(defn- get-tree
  "get an existing tree (non-recursive)"
  [request tree-sha]
  (go
    (if (= :missing-sha tree-sha)
      {}
      (<! (api-v3 (assoc request
                         :method client/get
                         :url #(gstring/format "https://api.github.com/repos/%s/%s/git/trees/%s"
                                               (:owner %)
                                               (:repo %)
                                               tree-sha)
                         :check-status (constantly true)))))))

(defn- get-commit
  [{:as request}]
  (api-v3 (assoc request
                 :url #(gstring/format "https://api.github.com/repos/%s/%s/git/commits/%s"
                                       (:owner %)
                                       (:repo %)
                                       (:sha %))
                 :method client/get
                 :check-status #(= 200 %))))

(defn- node-has-tree-path
  "if the node has a tree object, and the path matches then return the object"
  [node path]
  (some
   #(when (= path (:path %)) %)
   (let [[_ _ _ gh-object] node] (:tree gh-object))))

(defn- new-tree
  "create a new tree - response body will have a new sha for the tree content
   always makes one edit to an existing base tree (edit could be a blob or a tree - update or new)
   deletions not supported
   params
    - request
    - child node
    - parent node"
  [request [type path _ _ child] [_ _ _ base]]
  (log/infof "new tree %s -> %s %s %s" (->> base :tree (map :path)) type path (->> child :tree (map :path)))
  (api-v3 (assoc request
                 :url #(gstring/format "https://api.github.com/repos/%s/%s/git/trees"
                                       (:owner %)
                                       (:repo %))
                 :method client/post
                 :check-status (constantly true)
                 :body (merge {:tree [(case type
                                        :tree (assoc child
                                                     :mode "040000"
                                                     :path path
                                                     :type "tree")
                                        :blob {:path path
                                               :mode "100644"
                                               :type "blob"
                                               :sha (:sha child)})]}
                              (when (:sha base)
                                {:base_tree (:sha base)})))))

(defn converge-trees
  [request path content]
  (go-safe
   (let [[file & segments] (reverse (s/split path "/"))
          ;; construct a set of nodes from the root out to the new blob content
         nodes (cons
                [:tree :root ""]
                (reduce (fn [agg segment]
                          (cons [:tree segment ""] agg))
                        [[:blob file content]]
                        segments))
          ;; collect current trees out to blob
         current-nodes (<? (go-safe
                            (loop [to []
                                   from nodes]
                              (if-let [[type path :as node] (first from)]
                                (recur
                                 (conj
                                  to
                                  (case type
                                    :blob (conj node {})
                                    :tree (conj
                                           node
                                           (<? (get-tree
                                                request
                                                (cond
                                                       ;; root tree from commit
                                                  (= path :root)
                                                  (-> (<? (get-commit request)) :tree :sha)
                                                       ;; previous tree has an entry with this path so get the sha
                                                  (node-has-tree-path (last to) path)
                                                  (:sha (node-has-tree-path (last to) path))
                                                       ;; no tree entry
                                                  :else
                                                  :missing-sha))))))
                                 (rest from))
                                to))))]
     (println "current nodes")
     (pprint (->> current-nodes
                  (map #(let [[type path content current] %]
                          [type path content (-> current
                                                 (select-keys [:sha :tree])
                                                 (update-in [:tree] (fn [n] (->> n
                                                                                 (map :path)))))]))))
      ;; update back to root
     (<! (go
           (loop [last-node nil nodes (reverse current-nodes)]
             (if-let [[type _ content :as node] (first nodes)]
               (recur
                (case type
                  :blob (conj node (<! (new-blob request content)))
                  :tree (conj node (<! (new-tree request last-node node))))
                (rest nodes))
               (last last-node))))))))

(defn create-new-tree
  "walk updates and recursively create blobs, and updates trees"
  [request updates]
  (log/info "updates " @updates)
  (go
    (assoc
     request
     :atomist/tree
     (loop [tree nil f (seq @updates)]
       (if-let [[path content] (first f)] 
         (recur (<! (converge-trees request path content)) (rest f))
         tree)))))

(comment
  (go (println "final" (-> (<! (create-new-tree
                                {:ref {:owner "slimslenderslacks"
                                       :repo "segmentbuilder"
                                       :sha "11561c8a10bd5be167ef89b063b91dc93580a714"
                                       :branch "master"}
                                 :token (.. js/process -env -GITHUB_TOKEN_SLIMSLENDERSLACKS)}
                                {"path1/path2/crap.txt"
                                 "content"}))
                           :atomist/tree
                           (select-keys [:sha :tree])
                           (update-in [:tree] #(->> % (map :path)))))))
