(ns atomist.api
  (:require [atomist.json :as json]
            [clojure.tools.logging :as log]
            [atomist.topics :as topics]
            [atomist.graphql :as graphql]))

(defn- send-on-response-channel [o]
  (log/infof "publish message: %s" o)
  (topics/publish-json o))

(defn- default-destination [o]
  (if (or (not (:destinations o)) (empty? (:destinations o)))
    (-> o
        (update :destinations (constantly [(merge
                                            (:source o)
                                            {:user_agent "slack"})]))
        (update-in [:destinations 0] (fn [destination]
                                       (if (some? (:slack destination))
                                         (update destination :slack dissoc :user)
                                         destination))))
    o))

(defn success-status
  "on command request, send status that the invocation was successful"
  ([o status-message visibility]
   (-> (select-keys o [:topic :correlation-id :correlation_id :api_version :automation :team :command :source :destinations])
       (assoc :status (merge
                       {:code 0 :reason status-message}
                       (if visibility
                         {:visibility (name visibility)})))
       (assoc :content_type "application/x-atomist-status+json")
       (default-destination)
       (send-on-response-channel)))
  ([o]
   (success-status o "success" :info)))

(defn failed-status
  "on command request, send status that the invocation failed"
  ([o status-message]
   (-> (select-keys o [:topic :correlation-id :correlation_id :api_version :automation :team :command :source :destinations])
       (assoc :status {:code 1 :reason status-message})
       (assoc :content_type "application/x-atomist-status+json")
       (default-destination)
       (send-on-response-channel)))
  ([o]
   (failed-status o "failure")))

(defn event->request
  [request]
  (if (:extensions request)
    (log/debugf "handling event %s for %s/%s - %s" (->> request :data str (take 40) (apply str)) (-> request :extensions :team_id) (-> request :extensions :team_name) (-> request :extensions :correlation_id)))
  (cond-> request
    (:extensions request) (assoc :api_version "1"
                                 :correlation_id (-> request :extensions :correlation_id)
                                 :team {:id (-> request :extensions :team_id)
                                        :name (-> request :extensions :team_name)})))

(defn ^:api block-message
  "  params
       blocks - array of Slack blocks"
  [o blocks]
  (-> (select-keys o [:topic :correlation-id :correlation_id :api_version :automation :team :source :command :destinations :id :post_mode])
      (assoc :content_type "application/x-atomist-slack+json")
      (assoc :timestamp (System/currentTimeMillis))
      (assoc :body (-> {:text "fallback"
                        :blocks (json/->str blocks)}
                       (json/->str)))
      (default-destination)
      (send-on-response-channel)))

(defn add-destination [o {:keys [id name] :as chat-team} {:keys [id name] :as channel}]
  (update o :destinations (fnil conj []) {:user_agent "slack"
                                          :slack {:team chat-team
                                                  :channel channel}}))

(defn set-message-id [o id]
  (assoc o :id id))

(defn preview-url [request]
  (cond
    (= "https://automation.atomist.com/graphql" (:graphql-endpoint request)) "https://preview.atomist.com"
    (= "https://automation-staging.atomist.services/graphql" (:graphql-endpoint request)) "https://preview.atomist.services"
    :else "https://preview.atomist.com"))

(defn progress [request message]
  (letfn [(conj-if [coll {:keys [text] :as field}]
            (if (and text (not (= text "")))
              (conj coll field)
              coll))]
    (let [m (format "-----> %s" message)]
      (log/info m)
      (doseq [channel (or
                       (-> request :data :Push first :repo :channels)
                       (-> request :data :Tag first :commit :repo :channels))]
        (-> request
            (add-destination (:team channel) {:id (:channelId channel) :name (:name channel)})
            (set-message-id (format "%s:%s" (:repoId request) (:sha request)))
            (block-message [{:type "section"
                             :fields (-> []
                                         (conj-if {:type "plain_text" :text (:owner request)})
                                         (conj-if {:type "mrkdwn" :text (:repo request)})
                                         (conj-if {:type "mrkdwn" :text (-> request :skill-metadata :skill :name)})
                                         (conj-if {:type "mrkdwn" :text (-> request :skill-metadata :skill :version)})
                                         (conj-if {:type "mrkdwn" :text (format "*sha* %s" (:sha request))})
                                         (conj-if {:type "mrkdwn" :text (format "`%s`" (:correlation-id request))})
                                         (conj-if {:type "mrkdwn" :text (format "%s/log/%s/%s"
                                                                                (preview-url request)
                                                                                (:team-id request)
                                                                                (:correlation-id request))}))
                             :accessory {:type "image"
                                         :image_url "https://raw.githubusercontent.com/cljs/logo/master/cljs.png"
                                         :alt_text "cljs"}}
                            {:type "section"
                             :text {:type "mrkdwn"
                                    :text (format "```%s```" m)}}
                            {:type "context"
                             :elements [{:type "mrkdwn"
                                         :text "atomist-skills/package-cljs-skill version `0.2.15`"}]}]))))))

(defn container-reason [s]
  (log/info (format "atm:reason=%s" s)))

(defn status-message [request & {:keys [skipped success failure] :as status}]
  (cond
    failure
    (do
      (progress request (format "status message %s" status))
      (failed-status request failure)
      (container-reason failure))
    success
    (do
      (progress request (format "status message %s" status))
      (success-status request success nil)
      (container-reason success))
    skipped
    (do
      (success-status request skipped :hidden)
      (container-reason "skipped"))
    :else
    (do
      (failed-status request "status-message should be either success, failure, or skipped")
      (container-reason "status-message should be either success, failure, or skipped"))))

(defn get-repo-details [{:keys [owner repo branch] :as request}]
  (let [results (graphql/run-query request (graphql/get-branches-for-repo) {:owner owner :name repo})]
    {:sha (->> results
               :data
               :Repo
               first
               :branches
               (filter #(= branch (:name %)))
               first
               :commit
               :sha)
     :branchId (->> results
                    :data
                    :Repo
                    first
                    :branches
                    (filter #(= branch (:name %)))
                    first
                    :id)
     :repoId (-> results :data :Repo first :id)}))



