(ns atomist.functions.bootstrap
  (:require [clojure.tools.logging :as log]
            [atomist.api :as api]
            [atomist.functions.manage :as manage]
            [clojure.java.io :as io]
            [clojure.string :as s]
            [atomist.graphql :as graphql]
            [atomist.utils :as utils]
            [atomist.git :as git]
            [clojure.pprint :refer [pprint]]))

(defn process-namespace-in-name [m n]
  (assoc m
         :name (or (nth (re-find #"(.*)/(.*)" n) 2) n)
         :namespace (or (nth (re-find #"(.*)/(.*)" n) 1) "atomist")))

(defn filter-tags-from-readme [s]
  (let [[_ content] (re-find #"<!---atomist-skill-readme:start--->([\s\S]*)<!---atomist-skill-readme:end--->" s)]
    (or content s)))

(defn request->atomist-skill-input
  "request
     :subscriptions have already been pulled in from filesystem
     :skill, :parameters, and :commands have been pulled in from atomist.yaml
     :artifacts are ignored - we build the artifact section here"
  [request]
  (let [yaml (-> (:skill-metadata request)
                 (update-in [:artifacts :docker] (fn [coll] [(assoc (into {} (first coll)) :image (:docker-image request))]))
                 (assoc-in [:skill :version] (:version request)))
        skill (process-namespace-in-name (:skill yaml) (-> yaml :skill :name))
        atomist-skill-input (-> skill
                                (assoc :artifacts (:artifacts yaml))
                                ;; author
                                (assoc :branchId (:branchId request))
                                (assoc :categories (->> (or (:categories skill) (:category skill))
                                                        (map s/upper-case)))
                                (assoc :commands (-> yaml :commands))
                                (assoc :commitSha (:sha request))
                                ;; description
                                ;;(assoc :dispatchStyle (or (-> skill :dispatchStyle) (-> skill :dispatch)))
                                (assoc :displayName (or (-> skill :displayName) (-> skill :title) (-> request :skill-metadata :name)))
                                (assoc :homepageUrl (or (-> skill :homepageUrl) (-> skill :homepage)))
                                (assoc :iconUrl (or (-> skill :iconUrl) (-> skill :icon)))
                                ;; ingesters
                                ;; license
                                (assoc :longDescription (or
                                                         (-> skill :documentation)
                                                         (-> skill :longDescription)
                                                         (-> skill :description)
                                                         (-> skill :long_description)))
                                ;; name
                                ;; namespace
                                (assoc :parameters (-> yaml :parameters))
                                (merge
                                 (when-let [integrations (-> yaml :integrations)]
                                   {:integrations integrations}))
                                (merge
                                 (when-let [resource-providers (-> yaml :resourceProviders)]
                                   {:resourceProviders resource-providers}))
                                (assoc :readme (-> (slurp (io/file (:git-root request) "README.md"))
                                                   (filter-tags-from-readme)
                                                   (utils/base64)))
                                (assoc :repoId (:repoId request))
                                (assoc :subscriptions (-> yaml :subscriptions))
                                (assoc :technologies (->> (or (-> skill :technologies) (-> skill :technology))
                                                          (map s/upper-case)))
                                ;; version
                                (assoc :videoUrl (or (-> skill :videoUrl) (-> skill :video))))]
    (pprint atomist-skill-input)
    atomist-skill-input))

(defn register-skill [handler]
  (fn [request]
    (try
      (let [response (graphql/run-query
                      request
                      (graphql/register-skill)
                      {:skill (request->atomist-skill-input request)})]
        (log/infof "register-skill response body %s" response)
        (handler
         (assoc request
                :register-response
                response)))
      (catch Throwable t
        (.printStackTrace t)
        (log/error "register-response" (.getMessage t) (ex-data t))))))

(defn add-metadata [handler]
  (fn [request]
    (let [skill-metadata (merge
                          (manage/discover-skill-input (:git-root request))
                          (select-keys
                           (manage/parse-atomist-yaml (slurp (io/file (:git-root request) "atomist.yaml")))
                           [:artifacts :subscriptions]))]
      (handler (assoc request :skill-metadata skill-metadata)))))

(defn add-repo-details [handler]
  (fn [r]
    (let [request (assoc r
                         :branch "master"
                         :sha (git/sha r))]
      (handler request))))

(defn config [args]
  (if (= "prod" (second args))
    {:args (into [] args)
     :git-root (io/file (nth args 4))
     :api-key (first args)
     :branchId "T29E48P34_624642c2-a2c4-4cfb-b1b7-8752739dcfb9_atomist-skills_234407535_master"
     :repoId "T29E48P34_T29E48P34_atomist-skills_234407535"
     :team-id "T29E48P34"
     :graphql-endpoint "https://automation.atomist.com/graphql"
     :docker-image (nth args 2)
     :version (nth args 3)}
    {:args (into [] args)
     :git-root (io/file (nth args 4))
     :api-key (first args)
     :branchId "AK748NQC5_75b34148-c780-4cdb-a641-d4c34b74515e_atomist-skills_234407535_master"
     :repoId "AK748NQC5_AK748NQC5_atomist-skills_234407535"
     :team-id "AK748NQC5"
     :graphql-endpoint "https://automation-staging.atomist.services/graphql"
     :docker-image (nth args 2)
     :version (nth args 3)}))

(defn -main [& args]
  (try
    ((-> (fn [request] (log/info "success:  " (:register-response request)))
         (register-skill)
         (add-metadata)
         (add-repo-details)) (config args))
    (api/container-reason "finished")
    (System/exit 0)
    (catch Throwable t
      (.printStackTrace t)
      (log/error t)
      (System/exit 1))))
