(ns atomist.functions.manage
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [clojure.data]
            [digest]
            [atomist.utils :as utils])
  (:import (java.io File)))

(defn keywordize-path [x p]
  (if-let [path (get-in x [:event p])]
    (assoc-in x [:event p] (->> path (map #(cond (string? %) (keyword %) (number? %) (int %))) (into [])))
    x))

(defn keywordize-paths [x]
  (-> x
      (keywordize-path :path)
      (keywordize-path :team-path)
      (keywordize-path :chat-team-path)))

(defn subscription-operation-name [content]
  (->>
   content
   (re-find #"subscription\s*(\w*)\s*\{")
   second))

(defn subscription-name [content]
  (format "%s-%s"
          (subscription-operation-name content)
          (digest/sha-256 content)))

(def next-number (atom {}))
(defn get-next-name [s]
  (let [n (swap! next-number update s (fnil inc 0))]
    (format "%s%s" s (get n s))))

(defn edit-subscription-operation-name [{:keys [subscription] :as metadata}]
  (let [operation-name (subscription-operation-name subscription)
        replacement (get-next-name operation-name)
        m (-> metadata
              (assoc :name replacement)
              (assoc :subscription (clojure.string/replace-first subscription (re-pattern operation-name) replacement)))]
    m))

(defn discover-subscription
  "discover subscriptions from a graphql/subscription directory

   returns map with :subscriptions [{:keys [name description subscription]}] "
  [git-root]
  (let [subscription (File. git-root "graphql/subscription")]
    (if (.exists subscription)
      (let [subscriptions (->> (file-seq subscription)
                               (filter #(.isFile %))
                               (filter #(.endsWith (.getName %) "graphql"))
                               (map (fn [f] (let [content (slurp f)]
                                              {:name (subscription-name content)
                                               :description (format "stupendabot subscription for %s" subscription-name)
                                               :subscription content})))
                               (into [])
                               (doall))]
        {:subscriptions subscriptions}))))

(defn parse-atomist-yaml [s]
  (-> (->> (utils/yaml-parse-all s)
           (apply merge))
      (update :goals #(into [] %))))

(defn discover-skill-input
  [git-root]
  (let [catalog (io/file git-root "atomist.yaml")]
    (if (.exists catalog)
      (try
        (let [data (parse-atomist-yaml (slurp catalog))]
          {:skill (-> (into {} (:skill data))
                      (update-in [:category] (fn [categories] (if categories (into [] categories) [])))
                      (update-in [:technology] (fn [technologies] (if technologies (into [] technologies) []))))
           :parameters (:parameters data)
           :resourceProviders (:resourceProviders data)
           :commands (->> (:commands data)
                          (map #(if (contains? % :mapped_parameters)
                                  (assoc % :mapped_parameters (into [] (:mapped_parameters %)))
                                  %))
                          (map #(if (contains? % :secrets)
                                  (assoc % :secrets (into [] (:secrets %)))
                                  %))
                          (map #(if (contains? % :intent)
                                  (assoc % :intent (into [] (:intent %)))
                                  %))
                          (map #(if (contains? % :parameters)
                                  (assoc % :parameters (into [] (:parameters %)))
                                  %))
                          (into []))})
        (catch Throwable t
          (log/warn "Unable to parse atomist.yaml in %s" (.getName git-root))
          {})))))
