(ns atomist.topics
  (:require [atomist.json :as json]
            [clojure.tools.logging :as log])
  (:import (com.google.pubsub.v1 PubsubMessage)
           (com.google.cloud.pubsub.v1 Publisher)
           (com.google.api.core ApiFuture ApiFutures ApiFutureCallback)
           (com.google.protobuf ByteString)
           (java.util.concurrent Executors)))

(def publisher (atom nil))

(defn init-publisher [topic-name]
  (swap! publisher (constantly (.build (Publisher/newBuilder topic-name)))))

(defn shutdown-publisher []
  (Thread/sleep 5000)
  (.shutdown @publisher))

(defn publish [message callback]
  (if @publisher
    (let [message (.build (.setData (PubsubMessage/newBuilder) (ByteString/copyFromUtf8 message)))
          future (.publish @publisher message)]
      (ApiFutures/addCallback
       future
       (reify ApiFutureCallback
         (onSuccess [this s]
           (callback :success s))
         (onFailure [this t]
           (callback :failure t)))
       (Executors/newSingleThreadExecutor)))
    (log/info "publisher not initialized:  " message)))

(defn publish-json [obj]
  (publish
   (json/->str (dissoc obj :topic))
   (fn [status x]
     (case status
       :success (log/info "published:  " x)
       :failure (log/error x "failure publishing")
       (log/warn "bad status in publish callback")))))