(ns atomist.utils
  (:require [cheshire.core :as json]
            [clojure.string :as str :refer [blank? join]]
            [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [io.clj.logging :as mdc]
            [clj-time.core :as time]
            [schema.core :as s]
            [clojure.java.shell :as sh]
            [clj-yaml.core :as yaml]
            [clj-http.util :as util])
  (:import java.net.URL
           (java.io File)))

(defn log-with-mdc
  "Logs a message with mdc context"
  [level message mdc-map & [e]]
  {:pre [(not (blank? message))]}
  (mdc/with-logging-context mdc-map
    (if e
      (log/log level e message)
      (log/log level message))))

(defn warn-mdc
  "Warns with mdc context"
  [message mdc-map]
  (log-with-mdc :warn message mdc-map))

(defn info-mdc
  [message mdc-map]
  "Info with mdc context"
  (log-with-mdc :info message mdc-map))

(defn error-mdc
  [message mdc-map & [e]]
  "Error with mdc context"
  (log-with-mdc :error message mdc-map e))

(defn warn-mdc-trace
  "Warns with mdc context, adds a strack trace for debugging info"
  [message mdc-map]
  (try
    (throw (IllegalStateException. message))
    (catch Exception e
      (log-with-mdc :warn message mdc-map e))))

(defn info-mdc-trace
  "Info with mdc context, adds a strack trace for debugging info"
  [message mdc-map]
  (try
    (throw (IllegalStateException. message))
    (catch Exception e
      (log-with-mdc :info message mdc-map e))))

(defn warn-mdc-stacktrace-in-map
  "Warns with mdc context, adds a stack trace to the mdc map"
  ([message mdc-map]
   (try
     (throw (IllegalStateException. message))
     (catch Exception e
       (warn-mdc-stacktrace-in-map message mdc-map e))))
  ([message mdc-map e]
   (warn-mdc message (merge mdc-map {:stacktrace (join "\n" (map str (.getStackTrace e)))}))))

(defn info-mdc-stacktrace-in-map
  "Infos with mdc context, adds a stack trace to the mdc map"
  ([message mdc-map]
   (try
     (throw (IllegalStateException. message))
     (catch Exception e
       (info-mdc-stacktrace-in-map message mdc-map e))))
  ([message mdc-map e]
   (info-mdc message (merge mdc-map {:stacktrace (join "\n" (map str (.getStackTrace e)))}))))

(defn without-nils
  [m]
  (apply dissoc
         m
         (for [[k v] m :when (nil? v)] k)))

(defmacro log-time
  "Evaluate the expression, time it and log the time"
  [message & expr]
  `(let [start# (. System (nanoTime))
         ret# (do ~@expr)
         taken# (long (/ (double (- (. System (nanoTime)) start#)) 1000000.0))]
     (info-mdc (str ~message " [" taken# "]") {:duration taken#})
     ret#))

(defn assoc-if [m k v]
  (if v (assoc m k v) m))

(def not-blank-string (s/both s/Str (s/pred #(not (blank? %)) "string cannot be blank")))

(def not-blank-num-or-string
  (s/pred (fn [x] (cond (string? x) (not (blank? x))
                        (number? x) true
                        :else false))
          "Must be a non blank string or number"))

(defn url->base-url
  "Takes a github api url and returns the base github url or nil. Returns with a trailing slash."
  [url]
  (when-let [u (URL. url)]
    (str (.getProtocol u) "://" (.getHost u) "/")))

(defn string-or-false
  "Returns either the supplied string or false if the supplied string is blank"
  [s]
  (if-not (str/blank? s)
    s
    false))

(defn base64 [s]
  (util/base64-encode (.getBytes s)))

(defn load-edn-file
  [path]
  (-> path
      io/resource
      io/file
      slurp
      read-string))

(defn load-json-file
  [path]
  (-> path
      io/resource
      io/file
      slurp
      (json/parse-string true)))

(def ^:private duration-pattern
  #"^(?:(?<days>[0-9]+)d)(?:(?<hours>[0-9]+)h)?(?:(?<minutes>[0-9]+)m)?(?:(?<seconds>[0-9]+)s)?$")

(defn duration-to-millis
  [duration]
  (let [[days hours minutes seconds] (map (fn [s] (if s (Integer/parseInt s) 0)) (rest (re-matches duration-pattern duration)))]
    (+ (time/in-millis (time/days days))
       (time/in-millis (time/hours hours))
       (time/in-millis (time/minutes minutes))
       (time/in-millis (time/seconds seconds)))))

(defn mk-tmp-dir!
  "Creates a unique temporary directory on the filesystem. Typically in /tmp on
  *NIX systems. Returns a File object pointing to the new directory. Raises an
  exception if the directory couldn't be created after 10000 tries."
  []
  (let [base-dir (io/file (System/getProperty "java.io.tmpdir"))
        base-name (str (System/currentTimeMillis) "-" (long (rand 1000000000)) "-")
        tmp-base (File. (str base-dir "/" base-name))
        _ (.mkdirs tmp-base)
        max-attempts 10000]
    (loop [num-attempts 1]
      (if (= num-attempts max-attempts)
        (throw (Exception. (str "Failed to create temporary directory after " max-attempts " attempts.")))
        (let [tmp-dir-name (str tmp-base num-attempts)
              tmp-dir (io/file tmp-dir-name)]
          (if (.mkdir tmp-dir)
            tmp-dir
            (recur (inc num-attempts))))))))

(defn delete-directory-recursive
  "Recursively delete a directory."
  [^File file]
  ;; when `file` is a directory, list its entries and call this
  ;; function with each entry. can't `recur` here as it's not a tail
  ;; position, sadly. could cause a stack overflow for many entries?
  (when (.isDirectory file)
    (doseq [file-in-dir (.listFiles file)]
      (delete-directory-recursive file-in-dir)))
  ;; delete the file or directory. if it it's a file, it's easily
  ;; deletable. if it's a directory, we already have deleted all its
  ;; contents with the code above (remember?)
  (io/delete-file file))

(defn git-rev-parse [f]
  (->
   (sh/with-sh-dir
     f
     (sh/sh "git" "rev-parse" "HEAD"))
   :out
   (clojure.string/trim-newline)))

(defn capitalize-first-character [s]
  (apply str
         (let [letters (seq s)]
           (cons (clojure.string/capitalize (first letters)) (next letters)))))

(defn yaml-parse-all
  [^String string & {:keys [unsafe mark keywords] :or {keywords true}}]
  (for [doc (.loadAll (yaml/make-yaml :unsafe unsafe :mark mark) string)]
    (yaml/decode doc keywords)))
