(ns platform-order.core
  (:require [clj-time.core :refer [in-millis interval]]
            [clj-time.format :refer [parse]]
            [clj-time.local :refer [local-now]]
            [com.stuartsierra.component :as c]
            [platform-log.core :refer [info]]
            [platform-order.queue :refer [queue]]
            [platform-order.storage :refer [connect! disconnect!] :as storage]
            [schema.core :as s :refer [Any Num Keyword Str maybe optional-key validate make-fn-schema]]
            [utils.core :refer [uuid]]))

(def failure-limit 3)

(def Task {:meta-data {:id Str
                       :in-at Str
                       (optional-key :failures) Num
                       (optional-key :last-failed-at) Str
                       (optional-key :failure-message) Str}
           :func Keyword
           :args (maybe [Any])})

(defmulti do-task (fn [task & args] task))

(defmacro deftask
  "Ex: (deftask sample-job
         [{:keys [repository] :as system} a b]
         (+ a b))"
  [name bindings & body]
  `(defmethod do-task (keyword (str *ns*) (str '~name))
     [_# & args#]
     (let [~bindings args#]
       ~@body)))

(s/defn send!
  "Sends task to the queue. args must be data, no functions.
  Returns the task that was added if successful."
  [{:keys [storage-adapter] :as tasks}
   func :- Keyword
   & args]
  (let [id (uuid)
        t {:meta-data {:id id
                       :in-at (str (local-now))}
           :func func
           :args args}]
    (storage/add! storage-adapter t)
    (info id "Added task" func "to queue")
    t))

(s/defn complete!
  "Returns true if task completes successfully."
  [{:keys [meta-data func args] :as task} :- Task
   {:keys [storage-adapter] :as tasks}
   system]
  (let [{:keys [in-at id]} meta-data
        in-at (parse in-at)
        start-at (local-now)
        _ (apply (partial do-task func) (flatten [system args])) ; perform task
        end-at (local-now)]
    (info id "Completed task in" (in-millis (interval start-at end-at)) "ms"
      "(queued" (in-millis (interval in-at end-at)) "ms)")
    true))

(defn- attach-extra [message exception]
  (println exception)
  (if (= clojure.lang.ExceptionInfo (type exception))
    (assoc message :extra (ex-data exception))
    message))

(defn- send-to-back!
  [storage-adapter task]
  (storage/add! storage-adapter task))

(defn- send-to-failure-list!
  [storage-adapter task]
  (storage/fail! storage-adapter task))

(defn fail!
  "Assumes task is first in line of the queue. Queue must be an atom."
  [exception storage-adapter {:keys [meta-data func] :as task} {:keys [hard soft] :as exception-handlers}]
  (let [failures (+ 1 (or (:failures meta-data) 0))]
    (->> (update-in task [:meta-data] merge {:failures failures
                                             :last-failed-at (str (local-now))
                                             :failure-message (str exception)})
         (send-to-back! storage-adapter))
    (let [id (:id meta-data)]
      (if (>= failures failure-limit)
        (do
          (hard task exception) 
          (info id "Hard failing task" func))
        (do
          (soft task exception)
          (info id "Soft failing task" func))))
    true))

(defn clean-tasks
  "Returns non-failed tasks"
  [q]
  (filter #(let [f (-> % :meta-data :failures)]
             (or (nil? f)
                 (< f failure-limit))) q))

(defn work-next! [{:keys [storage-adapter] :as tasks} system exception-handlers]
  (when-let [{md :meta-data :as task} (storage/pop! storage-adapter)]
    (try
      (let [{:keys [failures]} md]
        (if (or (not failures)
                (< failures failure-limit))
          (complete! task tasks system)
          (send-to-failure-list! storage-adapter task)))
      (catch Exception e
        (fail! e storage-adapter task exception-handlers)))))

(defn work!
  "Takes messages off the queue, one by one, in order."
  [{:keys [tasks] :as system} exception-handlers]
  (info "Worker" "Ready to work...")
  (while true
    (Thread/sleep 1) ; keep cpu cycles low; probably room for improvement
    (work-next! tasks system exception-handlers)))

(def ExceptionHandler (make-fn-schema Any [[Task Any]]))
(def ExceptionHandlers {:soft ExceptionHandler
                        :hard ExceptionHandler})

(defrecord Worker [exception-handler tasks]
  c/Lifecycle
  (start [this]
    (assoc this :thread (future (work! this exception-handler))))

  (stop [this]
    (when-let [thread (:thread this)]
      (future-cancel thread))
    (assoc this :thread nil)))

(s/defn worker-component
  [exception-handlers :- ExceptionHandlers]
  (map->Worker {:exception-handler exception-handlers}))

(defrecord Tasks [storage-adapter jobs]
  c/Lifecycle
  (start [this]
    (connect! storage-adapter)
    this)

  (stop [this]
    (disconnect! storage-adapter)
    this))

(defn tasks-component [& [v]]
  (map->Tasks v))
