(ns com.beardandcode.forms.render
  (:require [clojure.string :as s]
            [clojure.set :refer [difference]]))

(defn- pick-title [name details]
  (let [title (details "title")]
    (if (nil? title)
      (s/capitalize (s/replace name #"[-_]" " "))
      title)))

(defn- password? [details]
  (= (details "format") "password"))

(defn- sort-properties [schema-properties ordered-names-raw]
  (let [all-names (set (keys schema-properties))
        ordered-names (filter all-names ordered-names-raw)
        unordered-names (difference all-names ordered-names)
        pick #(vector % (schema-properties %))]
    (concat (map pick ordered-names)
                    (map pick unordered-names))))


(defmulti render-property (fn [_ details _ _ _] (details "type")))


(defn schema
  ([schema-map enum-fns values errors] (schema schema-map enum-fns values errors []))
  ([schema-map enum-fns values errors prefixes]
    (map (fn [[name details]]
           (render-property (conj prefixes name) details enum-fns values errors))
         (sort-properties (schema-map "properties")
                          (schema-map "order")))))


(defn- find-value [values path]
  (get-in values path))

(defn- find-errors [errors path]
  (errors (str "/" (clojure.string/join "/" path))))

(defn- find-enum-fn [enum-fns path]
  (get-in enum-fns path))

(defn- as-id [path] (clojure.string/join "_" path))
(defn- as-name [path] (last path))

(defn error-list [errors]
  (map #(vector :p {:class "error"} %) errors))

(defmethod render-property "string" [path details enum-fns values errors]
  (let [prop-errors (find-errors errors path)
        id (as-id path)
        name (as-name path)
        existing-value (find-value values path)
        title (pick-title name details)
        body (concat (if (details "description") (list [:p (details "description")]) '())
             (error-list prop-errors)
             (list (if-let [enum-fn (find-enum-fn enum-fns path)]
                     [:select {:name id} [:option]
                      (for [entry (enum-fn)
                            :let [value (if (sequential? entry) (first entry) entry)
                                  label (if (sequential? entry) (second entry) entry)]]
                        [:option {:value value :selected (= value existing-value)} label])]
                     [:input {:type (if (password? details) "password" "text")
                              :name id
                              :value (if (password? details) nil existing-value)}])))
        error-class (if (> (count prop-errors) 0) "error" "")]
    (if (empty? title) body [:label {:class error-class :id id} title body])))

(defmethod render-property "object" [path details enum-fns values errors]
  [:fieldset {:id (as-id path)}
   (concat (list [:legend (pick-title (as-name path) details)])
           (error-list (find-errors errors path))
           (schema details enum-fns values errors path))])

(defn- num-display-fields [items details]
  (let [max-items (details "maxItems" 0)
        min-items (details "minItems" 0)
        num-items (count items)]
    (cond (> num-items max-items) num-items
          (> min-items 5) min-items
          (> max-items 0) max-items
          :else 5)))

(defmethod render-property "array" [path details enum-fns values errors]
  (let [item-details (assoc (details "items" {}) "title" "")
        items (find-value values path)
        item-values (if items
                      (->> items (map-indexed vector) (flatten)
                           (apply hash-map) (assoc-in values path))
                      values)]
    [:fieldset {:id (as-id path)}
     (concat (list [:legend (pick-title (as-name path) details)])
             (error-list (find-errors errors path))
             (list [:ol {:class "list-entry"
                         :data-item-title ((details "items" {}) "title")
                         :data-item-type (item-details "type")
                         :data-item-path (as-id path)
                         :data-min-items (details "minItems")
                         :data-max-items (details "maxItems")}
                    (for [i (range (num-display-fields items details))
                          :let [item-path (conj path i)
                                item-errors? (> (count (find-errors errors item-path)) 0)]]
                      [:li {:class (if item-errors? "error" "")}
                       (render-property item-path item-details enum-fns item-values errors)])]))]))

(defmethod render-property nil [path details _ values errors]
  (let [prop-errors (find-errors errors path)
        id (as-id path) name (as-name path)]
    (if-let [enum (details "enum")]
      [:fieldset {:class (if (> (count prop-errors) 0) "error" "") :id id}
       (concat (list [:legend (pick-title name details)])
               (if (details "description") (list [:p (details "description")]) '())
               (error-list prop-errors)
               (map #(vector :label
                             (let [input-attrs {:type "radio" :value % :name id}]
                               [:input (if (= (find-value values path) %)
                                         (assoc input-attrs :checked "checked")
                                         input-attrs)])
                             (s/capitalize %)) enum))])))

(defmethod render-property :default [& args]
  (println args))
