(ns com.beardandcode.forms
  (:require [hiccup.core :refer [html]]
            [com.beardandcode.forms.render :as render]
            [com.beardandcode.forms.schema :as schema]
            [com.beardandcode.forms.text :as text]))

(defmacro defschema [symbol path-or-hash]
  `(def ~symbol (schema/new ~path-or-hash)))

(def schema schema/new)

(defn build
  ([action schema] (build action schema {}))
  ([action schema {:keys [errors values enum-fns method csrf-fn error-text-fn]
                   :or {errors {}
                        values {}
                        enum-fns {}
                        method "POST"
                        csrf-fn (fn [& _] '())
                        error-text-fn (fn [_ _ error] (get text/english error "Unknown error"))}}]
     (let [schema-map (schema/as-map schema)
           error-text (reduce (fn [error-text [field field-errors]]
                                 (assoc error-text field (map #(error-text-fn schema field %) field-errors)))
                              {} errors)
           hiccup [:form {:action action :method method}
                   (concat '() (csrf-fn)
                           (render/error-list (error-text "/" []))
                           (render/schema schema-map enum-fns values error-text)
                           (list [:input {:type "submit" :value (schema-map "submit")}]))]]
       (html hiccup))))

(defn- with-prefix [n prefix]
  (if (empty? prefix)
    n (str prefix "_" n)))

(defn- values-using-schema [data schema-map key-fn prefix]
  (reduce (fn [out-map [name detail]]
            (let [name-with-prefix (with-prefix name prefix)]
              (case (detail "type")
                "object" (assoc out-map (key-fn name) (values-using-schema data detail key-fn name-with-prefix))
                "array" (let [value-pattern (re-pattern (str "^" name-with-prefix "_[0-9]+$"))
                              items (->> (keys data) (filter #(re-matches value-pattern %))
                                         (sort) (map data) (remove empty?))]
                          (if (empty? items) items (assoc out-map (key-fn name) items)))
                (if (and (contains? data name-with-prefix)
                         (not (empty? (data name-with-prefix))))
                  (assoc out-map (key-fn name) (data name-with-prefix))
                  out-map))))
          {} (schema-map "properties")))

(defn values [request schema & [{:keys [key-fn] :or {key-fn identity}}]]
  (values-using-schema (:form-params request) (schema/as-map schema) key-fn ""))

(defn errors
  ([request schema] (errors request schema {}))
  ([request schema {:keys [enum-fns csrf-field]
                    :or {enum-fns {}
                         csrf-field "__anti-forgery-token"}}]
   (let [params (->> (values request schema)
                     (filter (fn [[key value]] (not (= key csrf-field))))
                     (reduce concat [])
                     (apply hash-map))]
     (schema/validate schema params enum-fns))))
