(ns figwheel.main.schema.core
  (:require
   [clojure.java.io :as io]
   [clojure.string :as string]
   [clojure.spec.alpha :as s]
   [expound.printer :as printer]
   [expound.alpha :as exp]))

(def ^:dynamic *spec-meta* (atom {}))

(defn def-spec-meta [k & args]
  (assert (even? (count args)))
  (swap! *spec-meta* assoc k (assoc (into {} (map vec (partition 2 args)))
                                    :position (count (keys @*spec-meta*))
                                    :key k)))

(defn spec-doc [k doc] (swap! *spec-meta* assoc-in [k :doc] doc))

(defn file-exists? [s] (and s (.isFile (io/file s))))
(defn directory-exists? [s] (and s (.isDirectory (io/file s))))
(defn non-blank-string? [x] (and (string? x) (not (string/blank? x))))

;; ------------------------------------------------------------
;; Validate
;; ------------------------------------------------------------

(defn key-meta-for-problem [{:keys [via :spell-spec.alpha/likely-misspelling-of] :as prob}]
  (or
   (when-let [n (first likely-misspelling-of)]
     (when-let [ns (namespace (first via))]
       (get @*spec-meta* (keyword ns (name n)))))
   (some->> (reverse via)
            (filter @*spec-meta*)
            ;; don't show the root docs
            (filter (complement
                     #{:figwheel.main.schema.config/edn
                       :figwheel.main.schema.cljs-options/cljs-options}))
            first
            (get @*spec-meta*))))

(let [expected-str (deref #'exp/expected-str)]
  (defn expected-str-with-doc [_type spec-name val path problems opts]
    (str (expected-str _type spec-name val path problems opts)
         (when-let [{:keys [key doc]} (key-meta-for-problem (first problems))]
           (when doc
             (str
              "\n\n-- Doc for " (pr-str (keyword (name key)))  " -----\n\n"
             (printer/indent doc)))))))

(defn expound-string [spec form]
  (when-let [explain-data (s/explain-data spec form)]
    (with-redefs [exp/expected-str expected-str-with-doc]
      (with-out-str
        ((exp/custom-printer
          {:print-specs? false})
         explain-data)))))

(defn validate-config! [spec config-data context-msg]
  (if-let [explained (expound-string spec config-data)]
    (throw (ex-info (str context-msg "\n" explained)
                    {::error explained}))
    true))

#_(expound-string
   :figwheel.main.schema.config/edn
   (read-string (slurp "figwheel-main.edn")))

;; ------------------------------------------------------------
;; Generate docs
;; ------------------------------------------------------------

(defn markdown-option-docs [key-datas]
  (string/join
   "\n\n"
   (mapv (fn [{:keys [key doc]}]
           (let [k (keyword (name key))]
             (format "## %s\n\n%s" (pr-str k) doc)))
         key-datas)))

(defn markdown-docs []
  (let [{:keys [common un-common]} (->> (vals @*spec-meta*)
                                        (filter #(-> %
                                                     :key
                                                     namespace
                                                     (= "figwheel.main.schema.config")))
                                        (sort-by :position)
                                        (group-by :group))]
    (str "# Figwheel Main Configuration Options\n\n"
         "The following options can be supplied to `figwheel.main` via the `figwheel-main.edn` file.\n\n"
         "# Commonly used options (in order of importance)\n\n"
         (markdown-option-docs common)
         "\n\n"
         "# Rarely used options\n\n"
         (markdown-option-docs un-common))))

(defn output-docs [output-to]
  (require 'figwheel.main.schema.config)
  (require 'figwheel.server.ring)
  (.mkdirs (.getParentFile (io/file output-to)))
  (spit output-to (markdown-docs)))


#_(validate-config! :figwheel.main.schema.config/edn (read-string (slurp "figwheel-main.edn")) "")

#_(output-docs "doc/figwheel-main-options.md")

#_(markdown-docs)
