(ns ofcors.middleware
  (:require [clojure.string :as str]
            [cheshire.core :as json]))

;; wrap-exceptions: 
;; Middleware for handling exceptions in application responses.
;; It catches all exceptions and returns a JSON response with the exception type and message.
(defn wrap-exceptions [handler]
  (fn [request]
    (try
      (handler request)
      (catch Exception e
        (let [ex-type (-> e .getClass .getName)
              ex-message (.getMessage e)]
          {:status 500
           :headers {"Content-Type" "application/json"}
           :body (json/encode {:error ex-type
                               :message ex-message})})))))

;; wrap-preflight:
;; Middleware for handling CORS pre-flight requests (HTTP OPTIONS method).
;; It checks whether the request's origin is in the allowed origins list, and if
;; so, responds with the appropriate CORS headers, such as allowed methods and
;; allowed headers. If the origin is not allowed, it responds with a 403 status.
(defn wrap-preflight [handler cors-config]
  (fn [request]
    (if (= :options (:request-method request))
      (let [origin (get-in request [:headers "origin"])
            request-method (-> request
                               (get-in [:headers "access-control-request-method"])
                               (str/upper-case)
                               keyword)
            request-headers (->> request
                                 (get-in [:headers "access-control-request-headers"])
                                 (str/split #",")
                                 (map str/trim)
                                 (map str/lower-case))]
        (if (and (contains? cors-config :origins)
                 (contains? (:origins cors-config) origin))
          {:status 200
           :headers {"Access-Control-Allow-Origin" origin
                     "Access-Control-Allow-Methods" (->> (:methods cors-config)
                                                         (map name)
                                                         (map str/upper-case)
                                                         (str/join ","))
                     "Access-Control-Allow-Headers" (->> (:headers cors-config)
                                                         (map str/lower-case)
                                                         (str/join ","))}}
          {:status 403}))
      (handler request))))

;; wrap-cors:
;; Middleware for handling CORS headers in application responses.
;; It checks whether the request's origin is in the allowed origins list, and if
;; so, adds the "Access-Control-Allow-Origin" header with the request's origin.
;; If the allowed origins list is not provided, it adds the header with the value "*".
(defn wrap-cors [handler cors-config]
  (fn [request]
    (let [response (handler request)
          origin (get-in request [:headers "origin"])]
      (if (contains? cors-config :origins)
        (if (contains? (:origins cors-config) origin)
          (assoc-in response [:headers "Access-Control-Allow-Origin"] origin)
          response)
        (assoc-in response [:headers "Access-Control-Allow-Origin"] "*")))))

;; cors-middleware:
;; A combination of wrap-preflight and wrap-cors middleware.
;; It handles both CORS pre-flight requests and CORS headers in application responses.
(defn cors-middleware [handler cors-config]
  (-> handler
      (wrap-preflight cors-config)
      (wrap-cors cors-config)))


;; allow-all-origin-middleware:
;; Middleware for allowing all origins by setting the
;; "Access-Control-Allow-Origin" header to "*".
;; Use this function when you want to allow CORS requests from any origin.
(defn allow-all-origin-middleware [handler]
  (fn [request]
    (let [response (handler request)]
      (assoc-in response [:headers "Access-Control-Allow-Origin"] "*"))))

;; allow-same-origin-middleware:
;; Middleware for allowing only the same origin as the request by setting the
;; "Access-Control-Allow-Origin" header to the request's origin.
;; Use this function when you want to allow CORS requests only from the same
;; origin as the request itself.
(defn allow-same-origin-middleware [handler]
  (fn [request]
    (let [response (handler request)
          origin (get-in request [:headers "origin"])]
      (assoc-in response [:headers "Access-Control-Allow-Origin"] origin))))

;; allow-specific-origin-middleware:
;; Middleware for allowing specific origins by setting the
;; "Access-Control-Allow-Origin" header to the request's origin if it is
;; present in the allowed-origins list.
;; Use this function when you want to allow CORS requests only from a specific
;; set of origins.
(defn allow-specific-origin-middleware [allowed-origins]
  (fn [handler]
    (fn [request]
      (let [response (handler request)
            origin (get-in request [:headers "origin"])]
        (if (some #{origin} allowed-origins)
          (assoc-in response [:headers "Access-Control-Allow-Origin"] origin)
          response)))))