(defn rename-to-js
  "Rename any Clojure-based file to a JavaScript file."
  [file-str]
  (clojure.string/replace file-str #".clj\w*$" ".js"))

(defn relative-path
  "Given a directory and a file, return the relative path to the file
  from within this directory."
  [dir file]
  (.substring (.getAbsolutePath file)
              (inc (.length (.getAbsolutePath dir)))))

(defn js-file-name
  "Given a directory and file, return the relative path to the
  JavaScript file."
  [dir file]
  (rename-to-js (relative-path dir file)))

(defn sanitize
  "Replace hyphens with underscores."
  [s]
  (string/replace (name s) "-" "_"))

(def ^:dynamic *apps-base* "client/apps")

(defn starts-with-dot-or-hash [f]
  (or (.startsWith (.getName f) ".")
      (.startsWith (.getName f) "#")))

(defn filter-js-files [files]
  (filter (fn [f] (and (not (.isDirectory f))
                       (not (starts-with-dot-or-hash f))
                       (.endsWith (.getName f) ".js")))
          files))

(defn- copy-modified-js [js-sources]
  (println "copying modified js...")
  (println "js-sources: " (type js-sources))
  ;;(ensure-directory output-dir)
  (doseq [src (filter :modified? js-sources)]
    (.mkdirs (.getParentFile (:output-to src)))
    (io/copy (:file src) (:output-to src)))
  js-sources)

#_
(defn copy-to-public [files config]
  (let [apps-base (get-in config/config [:apps-dir])
        app-base (name (:project-name config))
        app-js-path (io/file apps-base app-base "js")
        apps-public (get-in config/config [:output-root])
        output-js-public (:generated-javascript config/config)]
    (->> (map (fn [f]
                (println "f is: " f)
                (assoc f :output-to
                      (io/file apps-public
                               output-js-public
                               (fs/relative-path app-js-path (:file f)))))
              files)
         copy-modified-js
         (map (fn [js'] {:file (:output-to js')})))))

(defn ns-marked-as-shared?
  "Is the namespace of the given file marked as shared?"
  ([jar file-name]
     (when-let [ns-decl (ns-find/read-ns-decl-from-jarfile-entry jar file-name)]
       (*shared-metadata* (meta (second ns-decl)))))
  ([file-name]
     (when-let [ns-decl (ns-file/read-file-ns-decl file-name)]
       (*shared-metadata* (meta (second ns-decl))))))

(defn shared-files-in-jars
  "Return all Clojure files in jars on the classpath which are marked
  as being shared."
  []
  (for [jar (classpath/classpath-jarfiles)
        file-name (ns-find/clojure-sources-in-jar jar)
        :when (ns-marked-as-shared? jar file-name)]
    {:js-file-name (rename-to-js file-name)
     :tag :cljs-shared-lib
     :compile? true
     :source (java.net.URL. (str "jar:file:" (.getName jar) "!/" file-name))}))

;; Overrides
(defn- replace-strings-with-files [watched-files]
  (map (fn [{:keys [source] :as m}]
         (assoc m :source (if (string? source) (io/file source) source)))
       watched-files))

(defn cljs-file?
  "Is the given file a ClojureScript file?"
  [f]
  (and (.isFile f)
       (.endsWith (.getName f) ".cljs")))

(defn clj-file?
  "Is the given file a ClojureScript file?"
  [f]
  (and (.isFile f)
       (.endsWith (.getName f) ".clj")))

(defn client-tools [& [options]]
  (let [tools-dir (io/file (name (:tools-dir config/config)))]
    (for [file (file-seq tools-dir)
          :when (and (not (.isDirectory file))
                     (not (starts-with-dot-or-hash file))
                     (or (ns-marked-as-shared? file) (cljs-file? file)))]
      (if (ns-marked-as-shared? file)
        {:js-file-name (js-file-name tools-dir file)
         :tag :cljs-shared
         :compile? true
         :source file}
        {:js-file-name (js-file-name tools-dir file)
         :tag :cljs
         :compile? true
         :source file}))))


(defn tag-modified-info [files config]
  (println "Tagging...")
  (let [cnt (count files)]
    (println cnt))
  (map
   (fn [f]
     (println f)
     (assoc f :modified?
            (> (.lastModified (io/file (or (:source f) (:file f))))
               (.lastModified
                (io/file (:output-to config))))))
   files))

(defn app-js-files [{:keys [project app output-dir] :as options}]
  (let [[project app] (map name [project app])
        js-dir (io/file project "src/js")
        app-js-dir (io/file js-dir project app)]
    (map (fn [f]
           {:file f
            :tag :js
            :output-to
            (io/file output-dir (fs/relative-path js-dir f))})
         (filter-js-files (file-seq js-dir)))))

(defn delete-js-file [options js-file-name]
  (let [js-file (io/file (str (:output-dir options) "/" js-file-name))]
    (when (.exists js-file) (.delete js-file))))

(defn force-compilation [options sources]
  (let [{:keys [tags triggers output-dir]} options]
    (when (and tags triggers)
      (let [res (reduce (fn [r t] (into r (get triggers t))) [] tags)
            files (filter (fn [x]
                            (some
                              (fn [re]
                                (re-matches re (:js-file-name x)))
                              res))
                          (filter #(= (:tag %) :cljs) sources))]
        (doseq [f files]
          (log/info :task :forcing :js-path (:js-file-name f))
          (delete-js-file options (:js-file-name f)))))))

(defn app-cljs-files [& [options]]
  (let [
        source-files (concat (doseq [source-path source-paths]
                               (file-seq source-path)))]
    (for [file source-files
          ;;_ (println file)
          :when (and (not (.isDirectory file))
                  (not (starts-with-dot-or-hash file))
                  (or (ns-marked-as-shared? file) (cljs-file? file)))]
      (if (ns-marked-as-shared? file)
        {:js-file-name (js-file-name project-cljs-dir file)
         :tag :cljs-shared
         :compile? true
         :source file}
        {:js-file-name (js-file-name project-cljs-dir file)
         :tag :cljs
         :compile? true
         :source file}))))

(defn build-js!
  [cljs-sources options]
  (println "building cljs sources...")
  (build
    (reify Compilable
      (-compile [_ options]
        (let [all-sources
              (flatten
                (map (fn [{:keys [js-file-name source]}]
                       (log/info :task :compiling :js-path js-file-name
                         :source source)
                       (-compile source
                         (assoc options :output-file js-file-name)))
                  cljs-sources))]
          (dependency-order all-sources))))
    options))

(defn filter-cljs-sources [cljs-sources options]
  (filter #(> (.lastModified (io/file (:source %)))
              (.lastModified (io/file (:output-to options))))
          cljs-sources))

(defn any-modified? [sources]
  (some :modified?  sources))

(defn forced-compilation-tags [cljs]
  (let [modifieds (filter :modified? cljs)]
    (when (not (empty? modifieds)) (set (distinct (map :tag modifieds))))))

;; for testing purposes
(defn my-sources [options]
  (concat (app-cljs-files options)
          (app-js-files options)
          (client-tools options)))

(defn- non-cljs-files [files]
  (map (fn [f] {:source f}) files))

(defn tagged-files-in-dir [dir tag ext]
  (map (fn [f] {:source f :tag tag})
       (filter #(.endsWith % ext)
               (map #(.getAbsolutePath %) (file-seq (io/file dir))))))

(defn html-files-in
  "Return a sequence of file maps for all HTML files in the given
  directory."
  ([dir]
     (html-files-in dir :html))
  ([dir tag]
     (tagged-files-in-dir dir tag ".html")))

(defn clj-files-in
  "Return a sequence of file maps for all Clojure files in the given
  directory."
  [dir tag]
  (tagged-files-in-dir dir tag ".clj"))

(defn get-watched-sources
  "Return source files where the :js-file-name does not match a
  regular expression in :ignore."
  [options sources]
  (if-let [ignore (:ignore options)]
    (remove (fn [src] (some #(and (:js-file-name src)
                                 (re-matches % (:js-file-name src)))
                           (map #(if (string? %) (re-pattern %) %) ignore)))
            sources)
    sources))

;; ================================================================
(defn output-dir
  "Determine the output dir."
  [config {:keys [app aspect]}]
  (str (or (get-in config [app :aspects aspect :output-dir])
         (:output-dir (:project config)))))

(defn output-file [config {:keys [app aspect]}]
  (get-in config [app :aspects aspect :output-file]))

(defn output-to [config app-aspect]
  (str (io/file (fs/ensure-directory (output-dir config app-aspect))
         (output-file config app-aspect))))

(defn cljs-compilation-options
  ([config {:keys [app aspect] :as app-aspect}]
     (assert app "cljs-compilation-opts requires an app name!")
     (let [aspect-cfg (get config [app :aspects aspect])
           build-opts (assoc (:build (app config))
                        :project (:name (:project config))
                        :app app
                        :output-dir (output-dir config app-aspect)
                        :libs (or (seq (:lib-paths (:project config)))
                                (:lib-paths config/config))
                        :output-file (output-file config app-aspect)
                        :output-to (output-to config app-aspect))]
       (merge (if-let [optimizations (:optimizations aspect-cfg)]
                (assoc build-opts :optimizations optimizations)
                build-opts)
              (:compiler-options aspect-cfg)))))
;; =====================================================================

;; TODO: tag modified info separately
(defn compile! [options]
  (println "options is " options)
  (let [classpath-sources (shared-files-in-jars)
        cljs-sources (app-cljs-files options)
        js-sources (app-js-files options)
        my-client-tools (client-tools options)
        my-sources (concat cljs-sources js-sources my-client-tools)
        watched-sources (tag-modified-info
                         (get-watched-sources options my-sources) options)]
    (println ".........................")
    (println "output-to:" (:output-to options))
    (when (or (any-modified? watched-sources) (:force? options))
      (do
        (println "compiling cljs...")
        (println "------------")
        (println "------------")
        (delete-deps-file options)
        (copy-modified-js (filter #(= :js (:tag %)) watched-sources))
        (build-js!
          (filter :compile? (concat watched-sources classpath-sources))
          (assoc options :tags (forced-compilation-tags watched-sources)))
        true))))

(defn compile-cljs
  ([config {:keys [app aspect] :as app-aspect}]
     (println "compiling cljs..")
     @(thread-safe-build!
        (cljs-compilation-options config app-aspect))))


;; ===========================================================-=======
;; Old devlopment server for cljs devlopment

(defn app-dev-servers []
  {:dev-app-servers (:dev-app-servers (:ports config/config))})

(defn- init-dev-server
  ([port project]
     (.mkdirs (io/file build/*tools-public*))
     (.mkdirs (io/file build/*public*))
     #_(assert (contains? config/config config-name)
         (str "Valid config names are " (pr-str (keys config/config)) "."))
     (if (not (app-development-servers* project))
       (let [config (fs/load-config project)
             port (or port (lease-port project :dev-app-servers))]
         (alter-var-root #'app-development-servers*
           assoc project (bservice/dev-service port config))
         (register-port :dev-app-servers port project :all)))))

(defn start-dev-server
  ([project]
     (start-dev-server nil project))
  ([port project]
     (init-dev-server port project)
     ((:start-fn (get app-development-servers* project)))
     {:status :ok
      :dev-app-servers (:dev-app-servers (:ports config/config))}))

(defn stop-dev-server
  "Stop the current application development server."
  [project]
  ((:stop-fn (get app-development-servers* project))))

(defn destroy-dev-server
  [project]
  (stop-dev-server project)
  (alter-var-root #'app-development-servers* dissoc project)
  (release-port :dev-app-servers project :all)
  {:dev-app-servers
   (vec (keys app-development-servers*))
   :ports
   (:dev-app-servers (:ports config/config))})

(defn restart-dev-server
 [project]
 (stop-dev-server project)
 (Thread/sleep 5000)
 (start-dev-server project))
