(ns bizlogic.tools.build
  (:require [config :as config]
            [bizlogic.tools.fs :as fs]
            [bizlogic.tools.utils :as utils]
            [cljs.analyzer :as ana]
            [cljs.closure :as cljsc
             :refer [build Compilable -compile ]]
            [cljs.js-deps :as jsdeps :refer [dependency-order]]
            [clojure.java.classpath :as classpath]
            [clojure.java.io :as io]
            [clojure.string :as string]
            [bizlogic.tools.log :as log]))

(defn delete-deps-file [opts]
  (let [output-dir (:output-dir opts)
        deps-file (io/file output-dir (:output-to opts))]
    (when (.exists deps-file)
      (.delete deps-file))))

(defn make-build-opts [prj-ctx app-ctx & [{:keys [mode] :or {mode :dev}
                                             :as opts}]]
  (println "opts:" opts)
  (let [public-root (:public-root prj-ctx)
        mode-options (get-in app-ctx [:mode mode])
        source-path (str (:cljs-path prj-ctx) "/" (name (:name app-ctx)))
        output-dir (:output-dir prj-ctx)
        output-to (str output-dir "/" (:output-to mode-options))
        source-map (:source-map mode-options)
        target (:target opts)
        optimizations (or (:optimizations mode-options) :none)]
    (merge {:app (:name app-ctx)
            :main (:main mode-options)
            :asset-path (:asset-path mode-options)
            :source-path source-path
            :output-to output-to
            :optimizations optimizations
            :libs (:libs prj-ctx)}
      (when (and (= optimizations :none) (not= target :nodejs))
        {:output-dir output-dir})
      (when source-map {:source-map source-map})
      (when target {:target target})
      opts)))

(defn build-app
  ([build-opts]
    (let [source-path (:source-path build-opts)]
      (println "build-opts is " build-opts)
      (println "-----------------")
      (println "compiling" (:app build-opts) "...")
      (println "-----------------")
      (delete-deps-file build-opts)
      (binding [] ;;
        (cljsc/build source-path build-opts))
      (println "finished compiling" (:app build-opts) "!")
      true))
  ([prj-ctx app-ctx opts]
    (let [mode-build-opts (make-build-opts prj-ctx app-ctx opts)]
      (build-app mode-build-opts))))

(defn mtime [f] (.lastModified (.getAbsoluteFile (io/file f))))

(defn some-newer? [files base]
  #_(println "files is:" files)
  (some #(> (mtime %) (mtime base)) files))

(defn build-once [prj-ctx app-ctx & [{:keys [mode] :as opts}]]
  (let [build-opts (make-build-opts prj-ctx app-ctx opts)
        files (file-seq (.getAbsoluteFile (io/file (:source-path build-opts))))
        base (:output-to build-opts)]
    (println "build-opts" build-opts)
    (println "files " files)
    (when (some-newer? files base)
      (build-app build-opts))))

(def ^:private build-agent (agent nil))

(defn thread-safe-build! [options]
  (println "In thread-safe-compile ..")
  (let [p (promise)]
    (send build-agent
      (fn [_] (deliver p
                (try
                  (println "agent-thread" (Thread/currentThread))
                  (build-app options)
                  (catch Throwable e
                    (do (log/error :exception e)
                        {:error (.getMessage e)}))))))
    p))
