(ns bizlogic.tools.reload
  (:require [clojure.string :as str]
            [cljs.core.async :as async])
  (:require-macros [cljs.core.async.macros :refer [go go-loop]]))

(defn data-identity [e]
  (.-data e))

(defn event-source!
  ([url-or-src]
   (event-source! url-or-src (async/chan 10) data-identity "message"))
  ([url-or-src ch]
   (event-source! url-or-src ch data-identity "message"))
  ([url-or-src ch f]
   (event-source! url-or-src ch f "message"))
  ([url-or-src ch f msg-type]
   (let [src (if (instance? js/EventSource url-or-src)
               url-or-src
               (js/EventSource. url-or-src))]
     (.addEventListener src
                        msg-type
                        (fn [e]
                          (async/put! ch (f e)))
                        false)
     {:source src
      :channel ch})))

(def sse-chan (async/chan 10))

(def sse-events (event-source! "/" sse-chan data-identity "count"))

(.log js/console "Starting to fetch events...")

(def evt-src (atom nil))

(defn start-reload-event []
  (let [pathname (.-pathname js/window.location)
        es (event-source!
             (str "/events/reload") sse-chan data-identity "reload")]
    (swap! evt-src (constantly es))
    (.log js/console "Starting event source..")
    (go-loop [event (async/<! sse-chan)]
      (.log js/console "New SSE event:" event)
      (if (= "reload" event)
        (do (.reload js/window.location true)
            (recur (async/<! sse-chan)))
        (.log js/console "There are no more events; Event channel is closed")))))

(start-reload-event)


(comment
  (async/go
    (async/>! build-chan
      (do
        (cljsc/build "tools/client/src/bizlogic/tools/reload/reload.cljs"
          {:output-to "public/global/javascripts/reload.js"
           :output-dir "public/global/javascripts"
           :optimizations :advanced
           :source-map "public/global/javascripts/reload.js.map"})
        :done)))
  )
