(ns bizlogic.tools.scheduler
  (:require ;;[io.pedestal.app.protocols :as p]
            [bizlogic.tools.log :as log]
            [bizlogic.tools.compile :as compile]
            [clojure.java.io :as io]
            [clojure.string :as string]
            [clojure-watch.core :as watcher]
            [clojurewerkz.quartzite.jobs :as qjobs :refer [defjob]]
            [clojurewerkz.quartzite.scheduler :as qs]
            [clojurewerkz.quartzite.triggers :as qt]
            [clojurewerkz.quartzite.schedule.simple :as simple]
            [clojurewerkz.quartzite.conversion :as qc])
  (:import (java.util.concurrent Callable Executors
                                 ScheduledExecutorService
                                 ScheduledFuture ThreadFactory
                                 TimeUnit)))

;; from Pedestal
(defprotocol Scheduler
  (schedule [this msecs f]
    "Schedules the no-argument function f to be called after N
    milliseconds. Returns a ScheduledTask.")
  (periodic [this msecs f]
    "Schedules the no-argument function f to be called repeatedly
    every N milliseconds. Returns a ScheduledTask which may be used to
    cancel all future invocations."))

(defprotocol ScheduledTask
  (cancel [this]
    "Cancels the scheduled task. Has no effect if the scheduled task
    has already occurred."))

(extend-type ScheduledFuture
  ScheduledTask
  (cancel [this] (.cancel this false)))

(extend-type ScheduledExecutorService
  Scheduler
  (schedule [this msecs f]
    (.schedule this ^Callable f msecs TimeUnit/MILLISECONDS))
  (periodic [this msecs f]
    (.scheduleAtFixedRate this f msecs msecs TimeUnit/MILLISECONDS)))

(defn scheduler []
  (Executors/newSingleThreadScheduledExecutor
   (reify ThreadFactory
     (newThread [this r]
       (doto (Thread. r "ScheduledExecutorService")
         (.setUncaughtExceptionHandler
          (reify Thread$UncaughtExceptionHandler
            (uncaughtException [this thread err]
              (log/error :exception err)))))))))


;; from bizlogic

(defn sanitize
  "Replace hyphens with underscores."
  [s]
  (string/replace s "-" "_"))

(defonce ^:private projects-context {})

(defn load-apps-config [project]
  (if (:apps-config (get projects-context project))
    (:apps-config (get projects-context project)))
  (let [project' (sanitize (name project))
        apps-config (load-file (str (io/file project' "config.clj")))]
    (alter-var-root #'projects-context
      #(assoc-in % [project :apps-config] apps-config))
    apps-config))

;; to run means to go from start to stop in a short range
(defmulti run-app-build (fn [arg] (type arg)) :default clojure.lang.Keyword)

(defmethod run-app-build clojure.lang.Keyword [prj-key]
  (run-app-build (load-apps-config prj-key)))

(defmethod run-app-build clojure.lang.PersistentArrayMap [app-ctx]
  (compile/build-app app-ctx))

(defjob AppBuildJob [job-ctx]
  (let [app-ctx (qc/from-job-data job-ctx)]
    (run-app-build app-ctx)))

(defn start-app-watch [{:keys [path callback]}]
  (watcher/start-watch
    [{:path path
      :event-types [:create :modify :delete]
      :bootstrap (fn [path] (println "Starting to watch" path))
      :callback callback}]))

(defn schedule-app-builder [scheduler prj-ctx app-ctx]
  (let [project (get-in prj-ctx [:project :name])
        app (:app app-ctx)
        _ (println "app is" app)
        prj-cljs-path (:cljs-path prj-ctx)
        app-cljs-path
        (str (io/file (name project) "src/cljs" (name project) (name app)))
        app-ctx (assoc app-ctx :source-path app-cljs-path)
        app-job-key (qjobs/key (str (name app) ".job") project)
        app-job-trigger-key (qt/key (str (name app) ".trigger") project)
        app-build-job (qjobs/build
                        (qjobs/of-type AppBuildJob)
                        (qjobs/using-job-data app-ctx)
                        (qjobs/with-identity app-job-key))
        app-build-trigger (qt/build
                            (qt/with-identity app-job-trigger-key)
                            ;;(qt/start-now)
                            )
        _ (println "app-cljs-path:" app-cljs-path)
        app-watcher-closer
        (if-not (:app-watcher-closer app-ctx)
          (start-app-watch
            {:path app-cljs-path
             :callback
             (fn [event filename]
               (println "Watch event" event "triggered for" filename)
               (run-app-build app-ctx))}))]
    (println "Scheduling ...")
    ;;(qs/schedule scheduler app-build-job app-build-trigger)
    (alter-var-root #'projects-context
      #(update-in % [project app] merge
         {:app-job-key app-job-key
          :app-job-trigger-key app-job-trigger-key
          :app-watcher-closer app-watcher-closer}))))

(defn- schedule-apps-builder [project]
  (let [scheduler (or nil (-> (qs/initialize) qs/start))
        apps-config (load-apps-config project)
        prj-ctx (select-keys apps-config [:project])
        apps (keys (dissoc apps-config :project))]
    (println "processing ...")
    (doseq [app apps]
      (let [app-ctx (assoc (get apps-config app) :app app)]
        (schedule-app-builder scheduler prj-ctx app-ctx)))))


(defn stop-app-watch [app-watcher-closer]
  (app-watcher-closer))

(defn de-schedule-app-builder [scheduler prj-ctx app-ctx]
  (let [app-build-trigger (get app-ctx :app-job-trigger-key)
        app-watcher-closer (get app-ctx :app-watcher-closer)]
    (qs/delete-trigger scheduler app-build-trigger)
    (stop-app-watch app-watcher-closer)))

;; un-schedule
(defn de-schedule-apps-builder [project]
  (let [scheduler (get projects-context [:scheduler])
        apps-config (get projects-context :apps-config)
        prj-ctx (get apps-config project)
        apps (keys (dissoc apps-config :project))]
    (doseq [app apps]
      (let [app-ctx (assoc (get apps-config app) :app app)]
        (de-schedule-app-builder scheduler prj-ctx app-ctx)))))
