(ns bizlogic.tools.session
  (:require [datomic.api :as d :refer [q db]]
            [ring.middleware.session.store :as store]
            [clj-uuid]
            [clj-time [core :as time] [coerce :as coerce]])
  (:import java.util.UUID))

(defonce session-uri "datomic:mem://session")

(d/create-database session-uri)

(defonce session-conn (d/connect session-uri))

(def session-schema  [{:db/id (d/tempid :db.part/db)
                       :db/ident :session/key
                       :db/valueType :db.type/uuid
                       :db/unique :db.unique/value
                       :db/cardinality :db.cardinality/one
                       :db/doc "Session Key"
                       :db.install/_attribute :db.part/db}
                      {:db/id (d/tempid :db.part/db)
                       :db/ident :csrf/anti-forgery-token
                       :db/valueType :db.type/string
                       :db/unique :db.unique/value
                       :db/cardinality :db.cardinality/one
                       :db/doc "CSRF Token"
                       :db.install/_attribute :db.part/db}])

@(d/transact session-conn session-schema)

(defn new-session-id []
  (clj-uuid/v4))

(defprotocol ISession
  (create-session [this user]
    "create session for user.")
  (delete-session [this key user-conn]
    "deletes a user's session from store."))

(defn session-store* [conn]
  (reify
    ISession
    (create-session [store user]
      (if-let [user-id (:user/id user)]
        (let [session-id (new-session-id)]
          @(d/transact conn
             [{:db/id #db/id[:db.part/user]
               :session/key session-id
               :user/id user-id}])
          {:session/key session-id})))

    (delete-session [store key users-conn]
      (let [[sid uid] (d/q '[:find [?e ?user-id]
                              :in $ ?key
                              :where
                              [?e :session/key ?key]
                              [?e :session/user ?user-id]]
                         (db conn) key)]
        (when uid
          (d/transact conn
            [[:db.fn/retractEntity sid]])
          (d/transact users-conn
            [{:db/id [:user/id uid]
              :session/last (time/now)}])
          sid)))

    store/SessionStore
    (read-session [store key]
      (println "session/key is: " key)
      (let [db (d/db conn)]
        (if-let [e (and (clj-uuid/uuid? (str key))
                     (d/entity db
                       (ffirst (d/q '[:find ?e
                                      :in $ ?uuid
                                      :where
                                      [?e :session/key ?uuid]]
                                 db (UUID/fromString (str key))))))]
          (d/touch e))))
    (write-session [store key session-data]
      (try
        (if-let [key (or (and (clj-uuid/uuid? (str key))
                           (UUID/fromString (str key))))]
          (do @(d/transact conn
                 [(merge {:db/id [:session/key key]}
                    session-data)])
              key))
        ;; TODO: complete with message
        (catch Exception e)))))

(defonce session-store (session-store* session-conn))
