(ns bizlogic.tools.middlewares
  (:require [bizlogic.tools.log :as log]
            [bizlogic.tools.utils :as btutils]
            [net.cgrand.enlive-html :as enlive]
            [io.pedestal.http.sse :as sse]
            [ring.util.mime-type :as mime]
            [ring.middleware.resource :as resource]
            [ring.middleware.file :as rmfile]
            [io.pedestal.interceptor.helpers :as interceptorh
             :refer [defon-request defbefore
                     defon-response defmiddleware]]
            [io.pedestal.impl.interceptor :as int-impl]
            [io.pedestal.http.route :as proute]
            [ring.middleware.edn :as edn]
            [ring.util.response :as ring-resp]
            [clojure.java.io :as io]
            [clojure.string :as string]
            [clojure.core.async :as async]
            [clojure-watch.core :as watcher])
  (:import java.io.File))

(defn render [s]
  (apply str s))

(defon-response default-cache-control-or-no-cache
  [response]
  (update-in response [:headers "Cache-Control"] #(or % "no-cache")))

(defn- edn-request?
  [req]
  (if-let [^String type (:content-type req)]
    (not (empty? (re-find #"^application/(vnd.+)?edn" type)))))

(defon-request edn-params
  [request]
  (if-let [body (and (edn-request? request) (:body request))]
    (let [edn-params (binding [*read-eval* false] (edn/-read-edn body))]
      (assoc request
        :edn-params edn-params
        :params (merge (:params request) edn-params)))
    request))

(defn file
  "Interceptor for file ring middleware."
  [root-path & [opts]]
  (let [root-path (str (io/file (io/resource root-path)))]
    (btutils/ensure-dir root-path)
    (interceptorh/handler ::file #(rmfile/file-request % root-path opts))))

(interceptorh/defon-request print-routes
  "Log the request's method and uri."
  [var request]
  (proute/print-routes (deref var))
  request)

#_
(defn after-router [routes-fn]
  (interceptorh/after
    ::after-router
    (fn [context]
      (let [routes (routes-fn)]
        (@#'proute/route-context-to-matcher-routes context
          (mapv #(assoc % :matcher (@#'proute/matcher %)) routes)
          routes)))))
;;=====================================================================

(interceptorh/defon-request log-request
  "Log the request's method and uri."
  [request]
  (log/info
    :msg "logging request without servlets"
    :request request)
  request)

(interceptorh/defon-request log-request-headers
  "Log the request's method and uri."
  [request]
  (log/info
    :msg (format "%s %s"
                         (string/upper-case (name (:request-method request)))
                         (:uri request))
    :headers (:headers request))
  request)

(defn log-request-key [keys]
  (interceptorh/on-request
    (fn [request]
      (let [key-path (symbol (string/join "/" keys))]
        (log/info :msg (str "logging request at " key-path)
          key-path (get-in request keys))))))

(interceptorh/defon-response log-response
  "Log the request's method and uri."
  [response]
  (log/info :msg "Final response map"
    :response response)
  response)

(interceptorh/defaround log-session
  ([ctx]
    (log/info :msg "session before"
      :session (get-in ctx [:request :session]))
    ctx)
  ([ctx]
    (log/info :msg "session after"
      :session (get-in ctx [:response :session]))
    ctx))

(interceptorh/defbefore log-context-before
  "log the context"
  [context]
  (log/info :msg "Logging the context"
    :context context)
  context)

(interceptorh/defafter log-context-after
  "log the context"
  [context]
  (log/info :msg "Logging the context"
    :context context)
  context)

(defn log-context-key [keys]
  (interceptorh/around
    (fn [ctx]
      (let [key-path (symbol (string/join "/" keys))]
        (log/info :msg (str "logging ctx at " key-path)
          key-path (get-in ctx keys))))
    (fn [ctx]
      (let [key-path (symbol (string/join "/" keys))]
        (log/info :msg (str "logging ctx at " key-path)
          key-path (get-in ctx keys))))))

(defn log-key-middleware [keys]
  (interceptorh/middleware
    ::log-key-middleware
    (fn [request]
      (let [key-path (symbol (string/join "/" keys))]
        (log/info :msg (str "logging request at " key-path)
          key-path (get-in request keys))
        request))
    (fn [response]
      (let [key-path (symbol (string/join "/" keys))]
        (log/info :msg (str "logging response at " key-path)
          key-path (get-in response keys))
        response))))

(defn- response?
  "A valid response is any map that includes an integer :status
  value."
  [resp]
  (and (map? resp)
       (integer? (:status resp))))

(interceptorh/defafter not-found
  "An interceptor that returns a 404 when routing failed to resolve a route."
  [context]
  (if-not (response? (:response context))
    (do (log/info :in :not-found :response (:response context))
        (assoc context
          :response (ring-resp/not-found (pr-str "Not Found!!"))))
    context))

(interceptorh/defbefore print-request
  [{:keys [request] :as ctx}]
  (let [intcpt (:name (peek (pop (::int-impl/stack ctx))))]
    (println (format "Printing request after Interceptor : %s " intcpt))
    (println request)
    ctx))


(defn file
  "Interceptor for file ring middleware."
  [root-path & [opts]]
  (interceptorh/handler ::file
    #(if-let [response (rmfile/file-request % root-path opts)]
       response
       %)))

;; oauth and session ================================================

(defn bearer-token [req])

(defn require-bearer-token [handler]
  (interceptorh/on-request
    (fn [request]
      (if-let [bearer-token (bearer-token request)]
        (assoc request :bearer-token bearer-token)
        request))))

(defn verify-bearer-token [& [find-token-fn]]
  (interceptorh/on-request
    (fn [{:keys [bearer-token] :as req}]
      )))

;; dev interceptors ======================================================

(defn content-type [response]
  (get-in response [:headers "Content-Type"]))

(defn add-event-source [{:keys [response request] :as context}]
  (println "In event-source-interceptor..")
  (let [body (:body response)
        uri (:uri request)]
    (println (str "request-uri is: " uri))
    (println (str "response-type is: " (content-type response)))
    (if (and body (or (= (content-type response) "text/html")
                    (= (content-type response) "text/html;charset=UTF-8")))
      (do (println "adding eventsource..")
          (update-in context [:response :body]
            (constantly
              (render
                (enlive/transform
                  (if (= (type body) String)
                    (enlive/html-snippet body)
                    (enlive/html-resource body))
                  [:body]
                  (enlive/append
                    (enlive/html-snippet
                      (str "<script type='text/javascript'"
                        "src='/javascripts/reload.js'></script>"))))))))
      context)))

(def event-source-interceptor
  (interceptorh/after ::event-source add-event-source))

;; watchers =============================================================

;; stored streaming channel
(defonce event-channel (atom nil))

(defn stream-channel-callback [chan _]
  (swap! event-channel
    #(do (if %1 (async/>!! %1 false)) chan)))

(defonce watchers (atom nil))

(defn html-watcher [config]
  (let [project (name (:name (:project config)))
        project-res-root (io/file project "resources" project)
        callback
        (fn [event filename]
          ;;(if (= filename path))
          (println (str "event is :" event))
          (println "Sending event to channel...")
          (if (= event :modify)
            (do (require (symbol (str project ".html")) :reload)
                (require (symbol (str project ".app")) :reload)
                (async/>!! @event-channel {:name "reload"
                                           :data "reload"}))))]
    (println (str "Starting html watcher for  " project))
    (swap! watchers #(do (if %1 (%1)) %2)
      (watcher/start-watch
        [{:path (str (io/file project-res-root "templates"))
          :event-types [:create :modify :delete]
          :bootstrap (fn [path] (println "Starting to watch " path))
          :callback callback}

         {:path (str (io/file project-res-root "public"))
          :event-types [:create :modify :delete]
          :bootstrap (fn [path] (println "Starting to watch " path))
          :callback callback}]))))

(defn start-html-watcher []
  (interceptorh/before
    (fn [{:keys [request config] :as context}]
      (println "Handling event sourcing...")
      (let [params (:params request)
            streamer-ctx (sse/start-stream
                           stream-channel-callback
                           context 10)]
        (if @event-channel
          (html-watcher config))
        streamer-ctx))))

(defbefore deref-async-builder
  [{:keys [request response builder config] :as ctx}]
  (println "deref'ing builder...")
  (if-let [chan (:channel builder)]
    (when-let [r (async/<!! chan)]
      (println (str "build-result is:" r))
      ctx)
    ctx))
