(ns doccla.oth-client.thresholds.patient-thresholds
  (:require [clj-http.client :as client]
            [doccla.oth-client.utils :as utils]
            [malli.clj-kondo :as clj-kondo]
            [malli.core :as m]
            [malli.instrument :as mi]
            [doccla.oth-client.schemas :as schemas]))

(def patient-thresholds-schema
  [:map
   [:total integer?]
   [:max integer?]
   [:offset integer?]
   [:results
    [:vector [:and schemas/threshold-schema
              [:map [:links
                     [:map
                      [:patient [:re utils/url-regex]]
                      [:patient-threshold [:re utils/url-regex]]
                      [:threshold-absolute [:re utils/url-regex]]]]]]]]])

(def query-params-schema
  [:map
   [:patient string?]])

(m/=> get-patient-thresholds [:=>
                              [:cat utils/opts-schema query-params-schema]
                              [:or utils/error-schema (utils/success-schema patient-thresholds-schema)]])

(defn post-processor
  [opts]
  (fn [data]
    (let [f (if (:validate-output? opts) m/coerce m/decode)]
      (f patient-thresholds-schema data utils/prune-map-transformer))))

(defn ^:mockable get-patient-thresholds
  "Retrieve the thresholds for a specific patient"
  [opts params]
  (let [url (->> "/thresholds/patient-thresholds"
                 (str (:base-url opts)))
        req (-> (utils/opts->request opts)
                (assoc :query-params params
                       :multi-param-style :array))
        res (client/get url req)]
    (utils/->output [200] (post-processor opts) res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
;; Configure linting
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example invocation
  (get-patient-thresholds {:base-url "https://doccla-dev.oth.io"
                           :validate-output? false
                           :auth {:type :id-secret
                                  :id "xxx"
                                  :secret "yyy"}}
                          {:patient "https://doccla-dev.oth.io/patients/37"}))
