(ns blueprint.helidon
  "An encoding of the most common API building case, allows
   building servers with a map of API command to handling
   function, leaving wiring to this"
  (:require [blueprint.handler :as handler]
            [clojure.tools.logging :as log]
            [exoscale.interceptor :as ix]
            [s-exp.hirundo :as http]))

(defn- start-callback
  [{::keys [options] :as this}]
  (log/info "Starting server with options" options)
  (assoc this ::server
         (http/start! (assoc options
                             :http-handler (handler/ring-handler this)
                             :legacy-return-value? false))))

(defn- stop-callback
  [{::keys [server] :as this}]
  (log/infof "Stopping server")
  (when (some? server)
    (http/stop! server))
  (dissoc this ::server))

(defn build
  "Builds a map describing a stopped API server, ready for use by
  `start` and `stop`.

  Acts as a component if wired to a system map."
  ([]
   (build {}))
  ([this]
   (handler/build this start-callback stop-callback ix/execute)))

(defn start
  "Start a server, expects the output of `build`, returns a modified
  map with an additional `:blueprint.helidon/server` key which
  holds the server.

  The `::options` keys is passed down directly to http-kit"
  [this]
  (handler/start this))

(defn stop
  "Stop a server, expects the output of `start`. Returns
   the input map without the `::server` key."
  [this]
  (handler/stop this))
