(ns exoscale.mania.config
  (:require [aero.core :as aero]
            [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [exoscale.coax :as cx]
            [clojure.string :as str]
            exoscale.mania.secret
            exoscale.mania.env))

(defmethod aero/reader 'split
  [_ _ value]
  (if (some? value)
    (str/split value #",")
    []))

(s/def :exoscale.mania.config/type #{:aero})
(s/def :exoscale.mania.config/reporter map?)
(s/def :exoscale.mania.config/logging map?)
(s/def :exoscale.mania.config/file string?)
(s/def :exoscale.mania.config/resource-file string?)

(defn throw!
  [e]
  (throw (ex-info "Error while loading configuration file"
                  {:type :exoscale.ex/incorrect}
                  e)))

(defn throw-config-not-found!
  [{::keys [file resource-file]}]
  (ex-info "Configuration file not found"
           {:type :exoscale.ex/not-found
            ::file file
            ::resource-file resource-file}))

(defn find-config
  "If one of resource-file or file should be specified, will try to load
  file first since it might come from the cli, then tries
  resource-file. At least one of the 2 must be reachable."
  [{::keys [file resource-file] :as options}]
  (or (try (some-> (or file (io/resource resource-file))
                   (doto slurp)) ;; just to make sure it exists and
                                 ;; has content, naive but works.
           (catch Exception _))
      (throw-config-not-found! options)))

(defmulti load! (fn [options] (:exoscale.mania.config/type options)))

(defmethod load! :aero
  [{:exoscale.mania.config/keys [spec profile resolver]
    :or {profile (some-> (System/getenv "PROFILE") keyword)}
    :as options}]
  (try
    (let [f (find-config (merge {::resource-file "config.edn"}
                                options))]
      (println (format "Loading config from %s" f))
      (->> (aero/read-config f (cond-> {}
                                 (some? profile)
                                 (assoc :profile profile)

                                 (some? resolver)
                                 (assoc :resolver resolver)))
           (cx/coerce spec)
           (s/assert* spec)))
    (catch Exception e
      (throw! e))))
