(ns exoscale.mania.component
  "Facilities to build a component system var"
  (:require [com.stuartsierra.component :as component]
            [clojure.spec.alpha         :as s]
            [exoscale.mania.reloadable  :as reloadable]))

(def reload-lock (Object.))

(defn- reloader
  [system]
  (component/update-system system (keys system) reloadable/reload))

(defn system-fns
  "Yield component system handling functions"
  [system-var builder]
  ;; Only validate the builder function, the rest gets validated later on
  (s/assert* ::builder builder)
  {::init   (fn [cfg] (alter-var-root system-var (constantly (builder cfg))))
   ::start  (fn []    (locking reload-lock (alter-var-root system-var component/start-system)))
   ::stop   (fn []    (locking reload-lock (alter-var-root system-var component/stop-system)))
   ::reload (fn []    (locking reload-lock (alter-var-root system-var reloader)))})
(defmacro defsystem
  "Define a var to hold a component system, and build functions
   to operate on it. The var will always be #'system in the current
   namespace.

   Yields a map of options ready to be used for `exoscale.mania/main-fn`."
  [{::keys [builder system-var init-var stop-var start-var reload-var] :as options}]
  (let [system-var (if (some? system-var) (symbol system-var) 'system)
        init-var   (if (some? init-var)   (symbol init-var)   'init!)
        stop-var   (if (some? stop-var)   (symbol stop-var)   'stop!)
        start-var  (if (some? start-var)  (symbol start-var)  'start!)
        reload-var (if (some? reload-var) (symbol reload-var) 'reload!)
        options    (dissoc options ::system-var ::init-var
                           ::stop-var ::start-var)]

    `(do
       ;; First we define a system var
       (defonce ~system-var nil)
       (alter-meta! (resolve '~system-var)
                    assoc :doc "Component system map" :redef true)

       ;; Then we build system manipulation functions
       (let [fns# (system-fns (resolve '~system-var) ~builder)]

         ;; Then we install them at the appropriate vars
         (def ~(or init-var   'init!)   "System init"  (::init fns#))
         (def ~(or stop-var   'stop!)   "System stop"  (::stop fns#))
         (def ~(or start-var  'start!)  "System start" (::start fns#))
         (def ~(or reload-var 'reload!) "System reload" (::reload fns#))

         ;; An updated options map is returned for def-main
         (assoc ~options
                :exoscale.mania.system/init   (::init fns#)
                :exoscale.mania.system/start  (::start fns#)
                :exoscale.mania.system/stop   (::stop fns#)
                :exoscale.mania.system/reload (::reload fns#))))))

(s/def ::builder ifn?)
