package com.flipkart.krystal.vajram.protobuf3.codegen;

import static com.flipkart.krystal.vajram.codegen.common.models.CodegenPhase.MODELS;
import static com.flipkart.krystal.vajram.protobuf3.codegen.Constants.MODELS_PROTO_FILE_SUFFIX;
import static com.flipkart.krystal.vajram.protobuf3.codegen.Constants.MODELS_PROTO_MSG_SUFFIX;
import static com.flipkart.krystal.vajram.protobuf3.codegen.Constants.VAJRAM_REQ_PROTO_FILE_SUFFIX;
import static com.flipkart.krystal.vajram.protobuf3.codegen.Constants.VAJRAM_REQ_PROTO_MSG_SUFFIX;
import static com.flipkart.krystal.vajram.protobuf3.codegen.Constants.VAJRAM_SVC_PROTO_FILE_SUFFIX;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ProtoGenUtils.capitalize;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ProtoGenUtils.createOutputDirectory;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ProtoGenUtils.getPackageName;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ProtoGenUtils.isProto3Applicable;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ProtoGenUtils.validateReturnTypeForProtobuf;
import static com.google.common.base.Preconditions.checkNotNull;

import com.flipkart.krystal.vajram.codegen.common.models.Utils;
import com.flipkart.krystal.vajram.codegen.common.models.VajramInfo;
import com.flipkart.krystal.vajram.codegen.common.models.VajramValidationException;
import com.flipkart.krystal.vajram.codegen.common.spi.AllVajramCodeGenContext;
import com.flipkart.krystal.vajram.codegen.common.spi.CodeGenerator;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;

/**
 * Code generator which generates protobuf schema for a request proto containing the input facets of
 * a vajram
 */
@Slf4j
class VajramServiceProto3SchemaGen implements CodeGenerator {

  private final AllVajramCodeGenContext creationContext;
  private final Utils util;

  public VajramServiceProto3SchemaGen(AllVajramCodeGenContext creationContext) {
    this.creationContext = creationContext;
    this.util = creationContext.util();
  }

  @Override
  public void generate() throws VajramValidationException {
    List<VajramInfo> applicableVajrams = getApplicable(creationContext, util);
    validate(applicableVajrams, util);
    generateServiceFiles(applicableVajrams);
  }

  private static List<VajramInfo> getApplicable(
      AllVajramCodeGenContext creationContext, Utils util) {
    if (!MODELS.equals(creationContext.codegenPhase())) {
      util.note("Skipping protobuf codegen since current phase is not MODELS");
      return List.of();
    }
    return creationContext.vajramInfos().stream()
        .filter(vajramInfo -> isProto3Applicable(vajramInfo, util))
        .toList();
  }

  /**
   * Validates the Vajram for protobuf compatibility. Throws exceptions if validations fail.
   *
   * @throws VajramValidationException if validation fails
   */
  static void validate(List<VajramInfo> vajramInfos, Utils util) throws VajramValidationException {
    for (VajramInfo vajramInfo : vajramInfos) {
      // Validate that the Vajram's return type conforms to protobuf RPC requirements
      validateReturnTypeForProtobuf(vajramInfo, util);
    }
  }

  /**
   * Generates the content for the service proto file that contains the service definition for the
   * remotely invocable Vajram.
   *
   * @param vajramInfos The Vajram information
   * @return The content for the service proto file
   */
  private void generateServiceFiles(List<VajramInfo> vajramInfos) {
    Map<String, List<VajramInfo>> vajramsByPackageName =
        vajramInfos.stream()
            .collect(Collectors.groupingBy(vajramInfo -> vajramInfo.lite().packageName()));

    try {
      // Create output directory if it doesn't exist
      Path outputDir = createOutputDirectory(util.detectSourceOutputPath(null), util);
      vajramsByPackageName.forEach(
          (packageName, vajramInfosForServiceName) -> {
            int lastDotIndex = packageName.lastIndexOf('.');
            String serviceName =
                capitalize(
                    lastDotIndex == -1 ? packageName : packageName.substring(lastDotIndex + 1));

            String serviceProtoFileName = serviceName + VAJRAM_SVC_PROTO_FILE_SUFFIX;
            StringBuilder protoBuilder = new StringBuilder();

            // Add auto-generated comment
            protoBuilder
                .append("// AUTOMATICALLY GENERATED - DO NOT EDIT!\n")
                .append("// This schema is auto-generated by Krystal's code generator.\n")
                .append("// It defines the service for Vajrams inside the package ")
                .append(packageName)
                .append("\n")
                .append("// Any manual edits to this file will be overwritten.\n\n");

            // Add syntax, package, and options
            protoBuilder.append("syntax = \"proto3\";\n\n");

            protoBuilder.append("package ").append(packageName).append(";\n\n");
            protoBuilder.append("option java_package = \"").append(packageName).append("\";\n");
            protoBuilder.append("option java_multiple_files = true;\n\n");
            // Add documentation for the service
            protoBuilder.append(
                "// Service definition for the remotely invocable Vajrams inside package %s\n"
                    .formatted(packageName));
            for (VajramInfo vajramInfo : vajramInfosForServiceName) {
              String vajramId = vajramInfo.vajramName();

              // Get the response type name (without package) to use in the service definition
              String responseTypeName =
                  ProtoGenUtils.getSimpleClassName(
                      vajramInfo.lite().responseType().canonicalClassName());

              // Add imports for the request and response messages
              protoBuilder
                  .append("import \"")
                  .append(vajramId)
                  .append(VAJRAM_REQ_PROTO_FILE_SUFFIX)
                  .append("\";\n");

              protoBuilder
                  .append("import \"")
                  .append(responseTypeName)
                  .append(MODELS_PROTO_FILE_SUFFIX)
                  .append("\";\n\n");
            }
            // Create the service definition
            protoBuilder.append("service ").append(serviceName).append(" {\n");
            for (VajramInfo vajramInfo : vajramInfosForServiceName) {
              String vajramId = vajramInfo.vajramName();

              // Get the response type name (without package) to use in the service definition
              String responseTypeName =
                  ProtoGenUtils.getSimpleClassName(
                      vajramInfo.lite().responseType().canonicalClassName());

              // Add the RPC method
              protoBuilder.append("  // Execute the Vajram remotely\n");
              protoBuilder
                  .append("  rpc Execute(")
                  .append(vajramId)
                  .append(VAJRAM_REQ_PROTO_MSG_SUFFIX)
                  .append(") returns (")
                  .append(
                      getPackageName(vajramInfo.lite().responseType().canonicalClassName())
                          .filter(
                              // We don't need to prefix messages with package
                              // names with the protos are in the same package
                              s -> !s.equals(packageName))
                          .map(s -> s + ".")
                          .orElse(""))
                  .append(responseTypeName)
                  .append(MODELS_PROTO_MSG_SUFFIX)
                  .append(");\n");
            }
            // Close the service definition
            protoBuilder.append("}\n");
            // Write service proto file
            Path serviceProtoFilePath = outputDir.resolve(serviceProtoFileName);
            log.info("Generated service protobuf schema file: {}", serviceProtoFilePath);

            try (PrintWriter out = new PrintWriter(Files.newBufferedWriter(serviceProtoFilePath))) {
              out.println(protoBuilder);
            } catch (IOException e) {
              util.error(
                  "Failed to generate service proto file %s".formatted(serviceProtoFilePath), null);
            }
          });

    } catch (IOException e) {
      util.error(
          String.format(
              "Error generating protobuf service definition for %s: %s",
              vajramInfos, e.getMessage()),
          null);
    }
  }
}
