(ns fluree.db.util.tx
  (:require [fluree.db.util.json :as json]
            [fluree.crypto :as crypto]
            [clojure.string :as str]
            [fluree.db.util.async :refer [<? go-try]]))

;; transaction utilities


(defn validate-command
  "Takes a command (map) and validates signature, adds in auth or authority and does
  some additional checks. This can be done before putting the command into the queue for processing.

  Puts original :cmd string and :sig string into this one map for use downstream."
  [{:keys [sig cmd]}]
  ;; TODO - here again we calc the sha3 id, I think redundant at this point
  (let [cmd-map       (-> (json/parse cmd)
                          (assoc :txid (crypto/sha3-256 cmd) ;; don't trust their id if provided
                                 :cmd cmd
                                 :sig sig))

        sig-authority (crypto/account-id-from-message cmd sig) ;; throws if invalid signature
        ;; merge everything together into one map for transaction.
        current-time  (System/currentTimeMillis)
        {:keys [auth authority expire]} cmd-map
        expired?      (and expire (< expire current-time))
        _             (when expired?
                        (throw (ex-info (format "Transaction is expired. Current time: %s expire time: %s." current-time expire)
                                        {:status 400 :error :db/invalid-transaction})))
        cmd-map*      (cond
                        (and (nil? auth) (nil? authority))
                        (assoc cmd-map :auth sig-authority)

                        (and (nil? auth) authority)
                        (throw (ex-info (str "An authority without an auth is not allowed.")
                                        {:status 400 :error :db/invalid-transaction}))

                        (and auth authority)
                        (if (= authority sig-authority)
                          cmd-map
                          (throw (ex-info (format "Signing authority: %s does not match command authority: %s." sig-authority authority)
                                          {:status 400 :error :db/invalid-transaction})))

                        (and auth (nil? authority))
                        (if (= auth sig-authority)
                          cmd-map
                          (assoc cmd-map :authority sig-authority)))]
    cmd-map*))


(defn get-tx-meta-from-tx
  "Separates tx-meta from the rest of the transaction.
  If by chance tx-meta was included twice, will throw an exception."
  [txn]
  (let [grouped (group-by #(if (str/starts-with? (:_id %) "_tx")
                             :tx-meta
                             :rest-tx) txn)
        tx-meta (when-let [tx-meta+ (not-empty (:tx-meta grouped))]
                  (when (not= 1 (count tx-meta+))
                    (throw (ex-info "You have multiple _tx metadata records in a single transaction, only one is allowed."
                                    {:status 400 :error :db/invalid-transaction})))
                  (->> tx-meta+
                       first
                       (reduce-kv (fn [acc k v]
                                    (cond
                                      (or (= :_id k) (= :_action k) (= :_meta k))
                                      (assoc acc k v)

                                      (nil? (namespace k))
                                      (assoc acc (keyword "_tx" (name k)) v)

                                      :else
                                      (assoc acc k v)))
                                  {})))]
    {:tx-meta tx-meta
     :rest-tx (:rest-tx grouped)}))


;(defn- add-nonce-to-tx-meta
;  "Only if a nonce was not included, adds one to the tx-meta.
;  This is needed if we are generating a default signature, to ensure
;  that the transactions Ids are unique.
;
;  To avoid this behavior, the user can always supply their own nonce id."
;  [tx-meta-txi]
;  (assoc tx-meta-txi :_id "_tx"                             ;; if tx-meta-txi is nil, make sure we give it an _id
;                     :_tx/nonce (or (:_tx/nonce tx-meta-txi) (:nonce tx-meta-txi) (System/currentTimeMillis))))

;
;(defn- get-db-default-private-key
;  "Will return nil if db doesn't have default auth configured, or we don't have the corresponding private key."
;  [conn db-ident default-auth-id]
;  (let [tx-key-id (:tx-key-id conn)]
;    (log/debug "Getting default private key for db." {:db db-ident :tx-key-id tx-key-id :default-auth-id default-auth-id})
;    (when (= tx-key-id default-auth-id)
;      (:tx-private-key conn))))

;
;(defn- get-default-auth-id
;  "If configured, returns the default auth id for the given database."
;  [db]
;  (go-try
;    ;; TODO - get rid of -subject interface, replace with query
;    (let [default-auth-subject (-> (<? (dbproto/-subject db ["_setting/id" "db"]))
;                                   (get "_setting/defaultAuth"))]
;      {:auth-subid (get default-auth-subject "_id")
;       :auth-id    (get default-auth-subject "_auth/id")})))
;
;
;(defn- generate-default-sig
;  "Looks up database to determine if we allow a default auth record
;  for unsigned transactions, and uses it to sign this transaction if available.
;  If not available, will throw an exception as it is valid."
;  [conn db-ident tx-str]
;  (go-try
;    (let [db          (<? (session/db conn db-ident nil))
;          {:keys [auth-subid auth-id]} (<? (get-default-auth-id db))
;          private-key (get-db-default-private-key conn db-ident auth-id)]
;      (when-not private-key
;        (throw (ex-info (str "Transaction does not have a signature, and database is not configured to accept a default auth (or this connection does not have access to the key).")
;                        {:status 400 :error :db/invalid-tx})))
;      {:sig-auth-id auth-subid :sig (crypto/sign-message tx-str private-key)})))

;
;(defn is-authority?
;  "Returns true if authority-auth is an authority for auth-ident"
;  [db authority-auth-ident auth-ident]
;  (go-try
;    (let [auth-subject       (<? (dbproto/-subject db auth-ident))
;          authority-auth-sid (<? (dbproto/-subid db authority-auth-ident))
;          authorities        (-> (get auth-subject "_auth/authority")
;                                 (map #(get % "_id")))]
;      (if (some #(= authority-auth-sid %) authorities)
;        true false))))

;
;(defn default-sign-tx
;  [conn db-ident tx-map]
;  "Attempts to default sign tx. Adds nonce to tx-meta if not already present.
;
;  Returns map of: :sig, :tx (as string), and :txid"
;  (let [{:keys [tx]} tx-map
;        txn      (if (string? tx)
;                   (json/parse tx)
;                   tx)
;        ;; separates tx-meta from the rest of the transaction
;        {:keys [tx-meta rest-txn]} (get-tx-meta-from-tx txn)
;        tx-meta* (add-nonce-to-tx-meta tx-meta)
;        tx-str*  (-> (conj rest-txn tx-meta*) (json/stringify))]
;    (if-let [priv-key (:tx-private-key conn)]
;      {:sig  (crypto/sign-message tx-str* priv-key)
;       :tx   tx-str*
;       :txid (crypto/sha3 tx-str*)}
;      (throw (ex-info "The transaction is not signed, and a default signing key is not present."
;                      {:status 400 :error :db/invalid-transaction})))))
;
;
