(ns com.flyingmachine.email.sending.send
  (:require [com.flyingmachine.email.core :as core]
            [com.flyingmachine.config :as config]
            [com.flyingmachine.email.sending.content :refer [body]]
            [clojure.java.io :as io]
            [clojure.string :as s]
            [stencil.core :as stencil]
            [clojure.contrib.core :refer (-?>)]
            [postal.core :as postal])
  (:import org.apache.commons.mail.HtmlEmail))


(defn params-or-config
  [params & keys]
  (or (get-in params key)
      (config/setting-in core/config-key key)))

(defn postal-metadata
  [params]
  {:host (config/setting core/config-key :hostname)
   :port (config/setting core/config-key :ssl-smtp-port)
   :username (params-or-config :authentication :username)
   :password (params-or-config :authentication :username)})

(defn postal-params
  [params]
  (merge params {:from (params-or-config :from-address)}))

(defn send-with-postal
  [params]
  (postal/send-message (postal-metadata params) (postal-params params)))

(defn send-with-apache-commons
  [params]
  (do
    (doto (HtmlEmail.)
      (.setHostName (config/setting core/config-key :hostname))
      (.setSslSmtpPort (config/setting core/config-key :ssl-smtp-port))
      (.setSSL (config/setting core/config-key :use-ssl))
      (.addTo (:to params))
      (.setFrom (params-or-config :from-address) (params-or-config :from-name))
      (.setSubject (:subject params))
      (.setTextMsg (get-in params [:body :text]))
      (.setHtmlMsg (get-in params [:body :html]))
      (.setAuthentication (params-or-config :authentication :username)
                          (params-or-config :authentication :password))
      (.send))
    true))

(defn send-email*
  [for-reals? params]
  (if for-reals?
    (cond
     (= (config/setting core/config-key :backend) :postal)
     (send-with-postal params)
     
     (= (config/setting core/config-key :backend) :apache-commons)
     (send-with-apache-commons params))
    params))

(defn send-email
  [params]
  (send-email* (config/setting :com.flyingmachine.email :send-email) params))

(defn final-sender-params
  [defaults addl template-name]
  (let [final (merge defaults addl)
        body-data (merge (:body-data defaults) (:body-data addl))]
    (-> final
        (merge {:body (list body template-name body-data)})
        (dissoc :body-data))))

(defn defsender
  [varnames sender-param-defaults sender]
  (let [{:keys [args user-doseq]} varnames
        [sender-name addl-args & sender-params] sender
        template-name (s/replace sender-name #"^send-" "")
        sender-params (final-sender-params sender-param-defaults
                                           (apply hash-map sender-params)
                                           template-name)
        args (into args addl-args)]
    
    `(defn ~sender-name
       ~args
       (doseq ~user-doseq
         (send-email ~sender-params)))))

(defmacro defsenders
  [varnames sender-param-defaults & senders]
  `(do ~@(map #(defsender varnames sender-param-defaults %) senders)))