(ns com.fulcrologic.rad.form-render
  "An extensible form generation system that uses styles to allow you to build generated forms that conform to patterns.

   Features:

   * Able to specify rendering as a default, specialized by style and/or specialized by entity type.
   * Subforms can additionally choose an alternate style based on context (ref target style on the relation)
   * Form fields can be composite: That is to say a group of related fields can be rendered as a cluster
     instead of having to be rendered separately.

  The renv is expected to be a form rendering-env, which contains things like:

  * ::form/master-form - The top-most form in the entire form tree
  * ::form/form-instance - Equivalent to this form
  * ::form/parent - when in a subform, this is the immediate parent
  * ::form/parent-relation - when in a subform, this is the ref key that was followed to get there.
  * ::attr/key->attribute - for at least the attributes appearing on the form set)

  "
  (:require
    [com.fulcrologic.fulcro.application :as app]
    [com.fulcrologic.fulcro.components :as comp]
    [com.fulcrologic.rad :as rad]
    [com.fulcrologic.rad.attributes :as attr]
    [com.fulcrologic.rad.attributes-options :as ao]
    [com.fulcrologic.rad.form-render-options :as fro]
    [com.fulcrologic.rad.form-options :as fo]
    [com.fulcrologic.rad.options-util :refer [?!]]
    [taoensso.timbre :as log]))

(defonce render-hierarchy (make-hierarchy))

(defmulti render-form
  "Render a form using the given environment. "
  (fn [renv id-attr]
    (let [style (or
                  (?! (fro/style id-attr) id-attr renv)
                  :default)]
      ;; NOTE: hierarchy maybe? (derive :sales/invoice :invoice/id). Make the rendering of :invoice/id forms be a
      ;; "kind" of :sales/invoice. Still need the possible dispatch to default style.
      [(ao/qualified-key id-attr) style]))
  :hierarchy (var render-hierarchy))

(defmulti render-header
  "Render a header for the given attribute."
  (fn [renv attr]
    (let [style (or
                  (?! (fro/header-style attr) attr renv)
                  :default)]
      [(ao/qualified-key attr) style]))
  :hierarchy (var render-hierarchy))

(defmulti render-fields
  "[rendering-env id-attribute]

   Render the fields of the current `form-instance` in `renv`."
  (fn [renv id-attr])
  :hierarchy (var render-hierarchy))

(defmulti render-footer
  "Render a header for the given attribute."
  (fn [renv attr]
    (let [style (or
                  (?! (fro/footer-style attr) attr renv)
                  :default)]
      [(ao/qualified-key attr) style]))
  :hierarchy (var render-hierarchy))



(defmulti render-field
  "Render a field. Should be installed as the render-field default for every form field type."
  (fn [{:com.fulcrologic.rad.form/keys [form-instance] :as renv} field-attr]
    (let [style (or
                  (?! (fro/style field-attr) (assoc renv ::context :field))
                  (?! (fo/field-style field-attr) form-instance)
                  (?! (ao/style field-attr) form-instance)
                  :default)]
      ;; not vector dispatch, because "default" type makes no sense
      [(ao/type field-attr) style]))
  :hierarchy (var render-hierarchy))



#_(defmethod render-field [::composite :default] [renv {::keys [children]}]
    (apply comp/fragment
      (map
        #(form/render-field renv %)
        children)))

#_(defn composite-field
    "Group attributes together such that `render-field` will be called with `unique-key` in order to render
     `children`."
    [unique-key children]
    {ao/qualified-key unique-key
     ao/type          ::composite
     ::children       children})

#_(defmethod render-field [:ref :default] [{::attr/keys [key->attribute]
                                          :com.fulcrologic.rad.form/keys [form-instance] :as renv} field-attr]
  (let [relation-key (ao/qualified-key field-attr)
        item         (-> form-instance comp/props relation-key)
        ItemForm     (-> form-instance fo/subforms fo/ui)
        to-many?     (= :many (ao/cardinality field-attr))]
    (render-header renv field-attr)
    (if to-many?
      (mapv (fn [i] (form/render-subform form-instance relation-key ItemForm i)) item)
      (form/render-subform form-instance relation-key ItemForm item))
    (render-footer renv field-attr)))

(defn derive!
  "Cause the given `child-keyword` to act as-if it were `parent-keyword` in the rendering multimethods. This
   does a `derive` on a custom hierarchy that is used for the rendering multimethods. The relation can be between
   styles, RAD attribute keys, etc.

   If you add your own multimethods you may choose to use `render-hierarchy` from this namespace to get all of these.
   e.g. `(defmulti sym (fn [] ...) :hierarchy gf/render-hierarchy)`. See Clojure documentation on multimethods
   and derive."
  [parent-keyword child-keyword]
  (derive render-hierarchy child-keyword parent-keyword))

(defn install-as!
  "Install the new form rendering support as a custom layout style `k`."
  [app k]
  (let [{::app/keys [runtime-atom]} app]
    (swap! runtime-atom assoc-in [::rad/controls :com.fulcrologic.rad.form/element->style->layout :form-container k]
      (fn [renv]
        (render-form renv (fo/id (comp/component-options (:com.fulcrologic.rad.form/form-instance renv))))))))
