(ns com.fulcrologic.fulcro.dom2
  "Client-side DOM macros and functions. For isomorphic (server) support, see also com.fulcrologic.fulcro.dom-server"
  (:refer-clojure :exclude [map meta time mask select use set symbol filter])
  (:require-macros [com.fulcrologic.fulcro.dom2])
  (:require
    [com.fulcrologic.fulcro.components :as comp]
    [react]
    [react-dom]
    [goog.object :as gobj]
    [com.fulcrologic.fulcro.dom-common :as cdom]))

(set! js/React react)
(set! js/ReactDOM react-dom)

(declare a abbr address altGlyph altGlyphDef altGlyphItem animate animateColor animateMotion animateTransform area
  article aside audio b base bdi bdo big blockquote body br button canvas caption circle cite clipPath code
  col colgroup color-profile cursor data datalist dd defs del desc details dfn dialog discard div dl dt
  ellipse em embed feBlend feColorMatrix feComponentTransfer feComposite feConvolveMatrix feDiffuseLighting
  feDisplacementMap feDistantLight feDropShadow feFlood feFuncA feFuncB feFuncG feFuncR feGaussianBlur
  feImage feMerge feMergeNode feMorphology feOffset fePointLight feSpecularLighting feSpotLight feTile feTurbulence
  fieldset figcaption figure filter font font-face font-face-format font-face-name font-face-src font-face-uri
  footer foreignObject form g glyph glyphRef h1 h2 h3 h4 h5 h6 hatch hatchpath head header hkern hr html
  i iframe image img input ins kbd keygen label legend li line linearGradient link main map mark marker mask
  menu menuitem mesh meshgradient meshpatch meshrow meta metadata meter missing-glyph
  mpath nav noscript object ol optgroup option output p param path pattern picture polygon polyline pre progress q radialGradient
  rect rp rt ruby s samp script section select set small solidcolor source span stop strong style sub summary
  sup svg switch symbol table tbody td text textPath textarea tfoot th thead time title tr track tref tspan
  u ul unknown use var video view vkern wbr)

(def ^{:private true} element-marker
  (-> (js/React.createElement "div" nil)
    (gobj/get "$$typeof")))

(defn element? "Returns true if the given arg is a react element."
  [x]
  (and (object? x) (= element-marker (gobj/get x "$$typeof"))))

(defn child->typed-child [child]
  (cond
    (string? child) [:string child]
    (number? child) [:number child]
    (or (vector? child) (seq? child) (array? child)) [:collection child]
    (nil? child) [:nil child]
    (element? child) [:element child]))

(defn parse-args
  "Runtime parsing of DOM tag arguments. Returns a map with keys :css, :attrs, and :children."
  [args]
  (letfn [(parse-css [[args result :as pair]]
            (let [arg (first args)]
              (if (keyword? arg)
                [(next args) (assoc result :css arg)]
                pair)))
          (parse-attrs [[args result :as pair]]
            (let [has-arg? (seq args)
                  arg      (first args)]
              (cond
                (and has-arg? (nil? arg)) [(next args) (assoc result :attrs [:nil nil])]
                (and (object? arg) (not (element? arg))) [(next args) (assoc result :attrs [:js-object arg])]
                (and (map? arg) (not (element? arg))) [(next args) (assoc result :attrs [:map arg])]
                :else pair)))
          (parse-children [[args result]]
            [nil (cond-> result
                   (seq args) (assoc :children (mapv child->typed-child args)))])]
    (-> [args {}]
      (parse-css)
      (parse-attrs)
      (parse-children)
      second)))

(defn render
  "Equivalent to React.render"
  [component el]
  (js/ReactDOM.render component el))

(defn render-to-str
  "Equivalent to React.renderToString. NOTE: You must make sure js/ReactDOMServer is defined (e.g. require cljsjs.react.dom.server) to use this function."
  [c]
  (js/ReactDOMServer.renderToString c))

(defn node
  "Returns the dom node associated with a component's React ref."
  ([component]
   (js/ReactDOM.findDOMNode component))
  ([component name]
   (some-> (.-refs component) (gobj/get name) (js/ReactDOM.findDOMNode))))

(defn create-element
  "Create a DOM element for which there exists no corresponding function.
   Useful to create DOM elements not included in React.DOM. Equivalent
   to calling `js/React.createElement`"
  ([tag]
   (create-element tag nil))
  ([tag opts]
   (js/React.createElement tag opts))
  ([tag opts & children]
   (js/React.createElement tag opts children)))

(defn convert-props
  "Given props, which can be nil, a js-obj or a clj map: returns a js object."
  [props]
  (cond
    (nil? props)
    #js {}
    (map? props)
    (clj->js props)
    :else
    props))

;; called from macro
;; react v16 is really picky, the old direct .children prop trick no longer works
(defn macro-create-element*
  "Used internally by the DOM element generation."
  [arr]
  {:pre [(array? arr)]}
  (.apply js/React.createElement nil arr))

(defn- arr-append* [arr x]
  (.push arr x)
  arr)

(defn- arr-append [arr tail]
  (reduce arr-append* arr tail))

;; fallback if the macro didn't do this
(defn macro-create-element
  "Runtime interpretation of props. Used internally by element generation when the macro cannot expand the element at compile time."
  ([type args] (macro-create-element type args nil))
  ([type args csskw]
   (let [[head & tail] (mapv comp/force-children args)]
     (cond
       (nil? head)
       (macro-create-element* (doto #js [type (cdom/add-kwprops-to-props #js {} csskw)]
                                (arr-append tail)))

       (element? head)
       (macro-create-element* (doto #js [type (cdom/add-kwprops-to-props #js {} csskw)]
                                (arr-append args)))

       (object? head)
       (macro-create-element* (doto #js [type (cdom/add-kwprops-to-props head csskw)]
                                (arr-append tail)))

       (map? head)
       (macro-create-element* (doto #js [type (clj->js (cdom/add-kwprops-to-props (cdom/interpret-classes head) csskw))]
                                (arr-append tail)))

       :else
       (macro-create-element* (doto #js [type (cdom/add-kwprops-to-props #js {} csskw)]
                                (arr-append args)))))))

(com.fulcrologic.fulcro.dom2/gen-client-dom-fns com.fulcrologic.fulcro.dom2/macro-create-element)
