(ns com.fulcrologic.fulcro.algorithms.react-interop
  #?(:cljs (:require-macros com.fulcrologic.fulcro.algorithms.react-interop))
  (:require
    #?(:cljs [cljsjs.react])
    [com.fulcrologic.fulcro.components :as comp :refer [defsc]]
    #?(:clj  [com.fulcrologic.fulcro.dom-server :as dom]
       :cljs [com.fulcrologic.fulcro.dom :as dom])
    [taoensso.encore :as enc]
    [taoensso.timbre :as log]))

(defn react-factory
  "Returns a factory for raw JS React classes.

  ```
  (def ui-thing (react-factory SomeReactLibComponent))

  ...
  (defsc X [_ _]
    (ui-thing {:value 1}))
  ```

  The returned function will accept CLJS maps as props (not optional) and then any number of children. The CLJS props
  will be converted to js for interop. You may pass js props as an optimization."
  [js-component-class]
  (fn [props & children]
    #?(:cljs
       (apply js/React.createElement
         js-component-class
         (dom/convert-props props)
         children))))

(defn react-input-factory
  "Returns a factory for raw JS React class that acts like an input. Use this on custom raw React controls are
  controlled via :value to make them behave properly with Fulcro.

  ```
  (def ui-thing (react-input-factory SomeInputComponent))

  ...
  (defsc X [_ _]
    (ui-thing {:value 1}))
  ```

  The returned function will accept CLJS maps as props (not optional) and then any number of children. The CLJS props
  will be converted to js for interop. You may pass js props as an optimization."
  [js-component-class]
  #?(:cljs
     (let [factory (dom/wrap-form-element js-component-class)]
       (fn [props & children]
         (apply factory (clj->js props) children)))
     :default
     (fn [props & children])))

(defn hoc-wrapper-factory
  "Creates a React factory `(fn [parent fulcro-props & children])` for a component that has had an HOC applied,
  and passes Fulcro's parent/props through to 'fulcro_hoc$parent' and 'fulcro_hoc_childprops' in the js props.

  See hoc-factory, which is more likely what you want, as it further wraps the parent context for proper interop."
  [component-class]
  (fn [this props & children]
    (when-not (comp/component? this)
      (log/error "The first argument to an HOC factory MUST be the parent component instance."))
    #?(:cljs
       (apply js/React.createElement
         component-class
         #js {"fulcro_hoc$parent"     this
              "fulcro_hoc$childprops" props}
         children))))

(defn hoc-factory
  "Returns a (fn [parent-component props & children] ...) that will render the target-fulcro-class, but as
  wrapped by the `hoc` function.

  Use this when you have a JS React pattern that tells you:

  ```
  var WrappedComponent = injectCrap(Component);
  ```

  where `injectCrap` is the `hoc` parameter to this function.

  Any injected data will appear as `:injected-props` (a js map) in the computed parameter of the target Fulcro component.

  You can this use the function returned from `hoc-factory` as a normal component factory in fulcro.
  "
  [target-fulcro-class hoc]
  (when-not (comp/component-class? target-fulcro-class)
    (log/error "hoc-factory MUST be used with a Fulcro Class"))
  (let [target-factory         (comp/computed-factory target-fulcro-class)
        target-factory-interop (fn [js-props]
                                 (let [parent       (comp/isoget js-props "fulcro_hoc$parent")
                                       fulcro-props (comp/isoget js-props "fulcro_hoc$childprops")]
                                   (comp/with-parent-context parent
                                     (target-factory fulcro-props {:injected-props js-props}))))
        factory                (let [WrappedComponent (hoc target-factory-interop)]
                                 (hoc-wrapper-factory WrappedComponent))]
    factory))

#?(:cljs (goog-define ERRORHEADER "Unexpected Error"))
#?(:cljs (goog-define ERRORMESSAGE "Please try again or contact support."))

(def *error-header* #?(:clj "Unexpected Error" :cljs ERRORHEADER))
(def *error-message* #?(:clj "Please try again or contact support." :cljs ERRORMESSAGE))

(defsc ErrorBoundary [this _props]
  {:shouldComponentUpdate    (fn [_np _ns] true)
   :getDerivedStateFromError (fn [error]
                               {:error true
                                :cause error})
   :componentDidCatch        (fn [_this error _info] (log/error (ex-message error)))}
  (let [{:keys [error cause]} (comp/get-state this)]
    (if error
      (dom/div :.ui.error.message
        (dom/div :.header *error-header*)
        (dom/p *error-message*)
        (dom/p (str cause))
        (when #?(:clj false :cljs goog.DEBUG)
          (dom/button {:onClick (fn []
                                  (comp/set-state! this {:error false
                                                         :cause nil}))} "Dev Mode: Retry rendering")))
      (comp/children this))))

(def ui-error-boundary (comp/factory ErrorBoundary))

#?(:clj
   (defn error-boundary* [[first-arg :as body]]
     `(let [real-parent# comp/*parent*]
        (ui-error-boundary {}
          (comp/with-parent-context real-parent#
            (comp/fragment ~@body))))))

#?(:clj
   (defn error-boundary-clj [body]
     `(try
        ~@body
        (catch Throwable _e#
          (dom/h1 *error-header*)
          (dom/h2 *error-message*)))))

#?(:clj
   (defmacro error-boundary
     "Wraps the given children in an ErrorBoundary stateless React component that prevents unexpected exceptions from
     propagating up the UI tree. Any unexpected rendering or lifecycle errors that happen *in the children* will be
     caught and cause an error message to be shown in place of the children. NOTE: Errors that occur at this
     level (i.e. the children you immediately wrap here) are NEVER caught (React limitation). This wrapper only protects
     you from errors that occur *within* the children.

     In other words:

     ```
     (error-boundary
       (throw (ex-info \"\" {})
       ...)
     ```

     will not catch the error, but it will catch any error caused *by* `ui-child` in:

     ```
     (error-boundary
       (ui-child props)
       ...)
     ```

     The header and message displayed in place of the children can be controlled by the Closure defines (in this namespace)
     ERRORHEADER and ERRORMESSAGE, or using the dynamic vars *error-header* and *error-message* from clj(s). Be careful
     when using the dynamic vars in React, since rendering can happen async in some cases. See `set!` to reset the root
     binding in cljs.
     "
     [& body]
     (if (enc/compiling-cljs?)
       (error-boundary* body)
       (error-boundary-clj body))))
