(ns com.fulcrologic.fulcro.inspect.element-picker
  (:require
    [clojure.string :as str]
    #?@(:cljs [[goog.object :as gobj]
               [goog.dom :as gdom]
               [goog.style :as gstyle]])
    [com.fulcrologic.fulcro.components :as fp :refer [defsc get-query]]))

(defn marker-element []
  (let [id "__fulcro_inspect_marker"]
    #?(:cljs
       (or (js/document.getElementById id)
         (doto (js/document.createElement "div")
           (gobj/set "id" id)
           (gobj/set "style" #js {:position       "absolute"
                                  :display        "none"
                                  :background     "rgba(67, 132, 208, 0.5)"
                                  :pointer-events "none"
                                  :overflow       "hidden"
                                  :color          "#fff"
                                  :padding        "3px 5px"
                                  :box-sizing     "border-box"
                                  :font-family    "monospace"
                                  :font-size      "12px"
                                  :z-index        "999999"})
           (->> (gdom/appendChild js/document.body)))))))

(defn marker-label-element []
  (let [id "__fulcro_inspect_marker_label"]
    #?(:cljs
       (or (js/document.getElementById id)
         (doto (js/document.createElement "div")
           (gobj/set "id" id)
           (gobj/set "style" #js {:position       "absolute"
                                  :display        "none"
                                  :pointer-events "none"
                                  :box-sizing     "border-box"
                                  :font-family    "sans-serif"
                                  :font-size      "10px"
                                  :z-index        "999999"
                                  :background     "#333740"
                                  :border-radius  "3px"
                                  :padding        "6px 8px"
                                  :color          "#ffab66"
                                  :font-weight    "bold"
                                  :white-space    "nowrap"})
           (->> (gdom/appendChild js/document.body)))))))

(defn react-raw-instance [node]
  #?(:cljs
     (if-let [instance-key (->> (gobj/getKeys node)
                             (filter #(str/starts-with? % "__reactInternalInstance$"))
                             (first))]
       (gobj/get node instance-key))))

(defn react-instance [node]
  #?(:cljs
     (if-let [raw (react-raw-instance node)]
       (or (gobj/getValueByKeys raw "_currentElement" "_owner" "_instance") ; react < 16
         (gobj/getValueByKeys raw "return" "stateNode")     ; react >= 16
         ))))

(defn pick-element [{::keys [on-pick]
                     :or    {on-pick identity}}]
  #?(:cljs
     (let [marker       (marker-element)
           marker-label (marker-label-element)
           current      (atom nil)
           over-handler (fn [e]
                          (let [target (.-target e)]
                            (loop [target target]
                              (if target
                                (if-let [instance (some-> target react-instance (fp/component-instance?))]
                                  (do
                                    (.stopPropagation e)
                                    (reset! current instance)
                                    (gdom/setTextContent marker-label (fp/component-name instance))

                                    (let [target' (js/ReactDOM.findDOMNode instance)
                                          offset  (gstyle/getPageOffset target')
                                          size    (gstyle/getSize target')]
                                      (gstyle/setStyle marker-label
                                        #js {:left (str (.-x offset) "px")
                                             :top  (str (- (.-y offset) 36) "px")})

                                      (gstyle/setStyle marker
                                        #js {:width  (str (.-width size) "px")
                                             :height (str (.-height size) "px")
                                             :left   (str (.-x offset) "px")
                                             :top    (str (.-y offset) "px")})))
                                  (recur (gdom/getParentElement target)))))))
           pick-handler (fn self []
                          (on-pick @current)

                          (gstyle/setStyle marker #js {:display "none"})
                          (gstyle/setStyle marker-label #js {:display "none"})

                          (js/removeEventListener "click" self)
                          (js/removeEventListener "mouseover" over-handler))]

       (gstyle/setStyle marker #js {:display "block"
                                    :top     "-100000px"
                                    :left    "-100000px"})

       (gstyle/setStyle marker-label #js {:display "block"
                                          :top     "-100000px"
                                          :left    "-100000px"})

       (js/addEventListener "mouseover" over-handler)

       (js/setTimeout
         #(js/addEventListener "click" pick-handler)
         10))))

(defn inspect-component [comp]
  {:fulcro.inspect.ui.element/display-name (fp/component-name comp)
   :fulcro.inspect.ui.element/props        (fp/props comp)
   :fulcro.inspect.ui.element/ident        (try
                                             (fp/get-ident comp)
                                             (catch :default _ nil))
   :fulcro.inspect.ui.element/query        (try
                                             (some-> comp fp/react-type fp/get-query)
                                             (catch :default _ nil))})
