# shrubbery

A stubbing, spying, and mocking library for Clojure protocols.

## Purpose

Clojure protocols are a great way to encapsulate operations with side effects, but suffer from a lack of general
test tooling. Shrubbery provides a small set of basic building blocks for working with them:

 * `stub`, which accepts a protocol and a hashmap of functions and returns an implementation with those functions
    (functions without implementations return `nil`);
 * `spy`, which accepts a protocol and some implementation and returns a new implementation that tracks the number of 
    times each of its members were called;
 * `received?`, which in conjunction with the `Matcher` protocol provides a way to query spies programmatically; and
 * `mock`, which wraps a `stub` in a `spy`, allowing callers to supply basic function implementations _and_ assert
   against those calls

Both spies and stubs can be used more or less independently; any protocol implementation may be wrapped by a spy, and 
stubs need not be spies to provide basic utility.

Shrubbery is test framework-agnostic, has no external dependencies, makes no attempt to perform runtime var replacement
and does not add any unusual syntax. It is simply meant to provide basic protocol implementation support in the meagre 
hope that it makes your day slightly more pleasant.

### Before You Begin

If you aren't already using a library like [Component](https://github.com/stuartsierra/component) to structure your 
Clojure application, or do not otherwise make much use of protocols, you may not find this library very interesting. 
Additionally, for more background on mock objects, you may find it helpful to read Martin Fowler's classic article on
the subject, [Mocks Aren't Stubs](http://martinfowler.com/articles/mocksArentStubs.html).

## Usage

### Stubs

Stubs are essentially sugar over `reify`, allowing convenient programmatic access to protocol creation. By default,
a stub without implementations simply returns `nil` for all method calls rather than throwing an
`IllegalArgumentException`. Optionally, return values for some or all members of the protocol may be provided using a 
hashmap of method-name to value.

Shrubbery attempts to detect whether or not the implementation it has been given is already a function or not. If it
is, that function is used unaltered; plain objects simply become the return value for that method.

Once defined, stubs cannot currently be altered, though they may be introspected; see the `Stub` protocol.

```clojure
(ns example
  (:require [clojure.test :refer :all]
            [shrubbery.core :refer :all]))

(defprotocol DbClient
  (select [t sql]))

(defn find-user [client id]
  (select client (str "select * from users where id = " id)))

(deftest test-find-user
  (let [subject (stub DbClient)]
    (is (nil? (find-user subject 42))))
  (let [subject (stub DbClient {:select "wow"})]
    (is (= "wow" (find-user subject 42))))
  (let [subject (stub DbClient {:select (fn [_ sql] (str "sql was " sql)})]
    (is (= "sql was select * from users where id = 42"
           (find-user subject 42))))
  )
```

### Spies

A test spy is an object that understands how many times its members have been called, and tracks the arguments those
members were called with. Shrubbery spies take a protocol and some pre-existing implementation and delegates all calls 
to the underlying implementation. Note that Shrubbery provides no attempt to _automatically_ verify call counts, as 
with classic mocking frameworks: currently, you as the test author are expected to perform your own assertions against
the spy at the end of the test.

Spies are queries directly use `calls`, which simply returns a hashmap of all calls, or indirectly using `call-count` 
and `received?`, both of which allow you to filter the count. Each function comes in two forms: given a
spy and a keyword naming a method, it returns the appropriate calls; if additionally given a vector of expected 
arguments, it will limit the query to calls matching those arguments.

Special support is provided for querying arguments against certain objects via the `Matcher` protocol; in particular, 
note that regular expressions are treated as if they are intended to be matched against strings rather than checked 
for direct object equality. This is useful in cases where you might want to make a rough assertion against a string's
contents but don't care about an exact match.

As with stubs, once defined, spies cannot currently be altered, though they may be introspected; see the `Spy` protocol.
 
```clojure
(ns example
  (:require [clojure.test :refer :all]
            [shrubbery.core :refer :all]))

(defprotocol DbClient
  (select [t sql]))
  
(def fake-db-client
  (reify DbClient
    (select [t sql] {:id 1})))
    
(defn find-user [client id]
  (select client ...))

(deftest test-find-user
  (let [subject (spy DbClient fake-db-client)]
    (is (not (received? subject :select)))
    (is (= {:id 1} (find-user subject 42)))
    (is (received? subject :select))
    (is (received? subject :select [42]))
    ))
```

### Mocks

Mocks combine the call count capabilities of a spy with the sugar of stubs. They act like a stub for the purposes of
initial definition but act like a spy for functions like `call-count` and `received?`. More precisely, they first
call `stub` with the given arguments, then wrap that stub in a `spy`.

```clojure
(ns example
  (:require [clojure.test :refer :all]
            [shrubbery.core :refer :all]))

(defprotocol DbClient
  (select [t sql]))

(defn find-user [client id]
  (select client ...))

(deftest test-find-user
  (let [subject (mock DbClient {:select {:id 1}})]
    (is (not (received? subject :select)))
    (is (= {:id 1} (find-user subject 42)))
    (is (received? subject :select))
    (is (received? subject :select [42]))
    ))
```

## License

![Then when you have found the shrubbery](https://31.media.tumblr.com/e72f365e1656130bbaebd2a2431c958b/tumblr_nia9ciTmpj1u0k6deo4_250.gif)

Copyright © 2015 Brian Guthrie

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
