(ns clojure-lsp.feature.diagnostics
  (:require
   [clj-kondo.impl.config :as kondo.config]
   [clojure-lsp.logger :as logger]
   [clojure-lsp.queries :as q]
   [clojure-lsp.settings :as settings]
   [clojure-lsp.shared :as shared]
   [clojure.core.async :as async]
   [clojure.java.io :as io]))

(set! *warn-on-reflection* true)

(def diagnostic-types-of-unnecessary-type
  #{:clojure-lsp/unused-public-var
    :redefined-var
    :redundant-do
    :redundant-expression
    :redundant-let
    :unused-binding
    :unreachable-code
    :unused-import
    :unused-namespace
    :unused-private-var
    :unused-referred-var})

(def deprecated-diagnostic-types
  #{:deprecated-var})

(defn ^:private kondo-config-for-ns [kondo-config ns-name]
  (let [ns-group (kondo.config/ns-group kondo-config ns-name)
        config-in-ns (get (:config-in-ns kondo-config) ns-group)
        kondo-config (if config-in-ns
                       (kondo.config/merge-config! kondo-config config-in-ns)
                       kondo-config)]
    kondo-config))

(defn ^:private unused-public-var->finding [element kondo-config]
  (let [keyword-def? (identical? :keyword-definitions (:bucket element))
        kondo-config (if (:ns element)
                       (kondo-config-for-ns kondo-config (:ns element))
                       kondo-config)]
    {:filename (:filename element)
     :row (:name-row element)
     :col (:name-col element)
     :end-row (:name-end-row element)
     :end-col (:name-end-col element)
     :level (or (-> kondo-config :linters :clojure-lsp/unused-public-var :level) :info)
     :message (if keyword-def?
                (if (:ns element)
                  (format "Unused public keyword ':%s/%s'" (:ns element) (:name element))
                  (format "Unused public keyword ':%s'" (:name element)))
                (format "Unused public var '%s/%s'" (:ns element) (:name element)))
     :type :clojure-lsp/unused-public-var}))

(defn ^:private exclude-public-diagnostic-definition? [kondo-config definition]
  (let [kondo-config (kondo-config-for-ns kondo-config (:ns definition))
        excluded-syms-regex (get-in kondo-config [:linters :clojure-lsp/unused-public-var :exclude-regex] #{})
        excluded-defined-by-syms-regex (get-in kondo-config [:linters :clojure-lsp/unused-public-var :exclude-when-defined-by-regex] #{})
        fqsn (symbol (-> definition :ns str) (-> definition :name str))]
    (or (q/exclude-public-definition? kondo-config definition)
        (some #(re-matches (re-pattern (str %)) (str fqsn)) excluded-syms-regex)
        (some #(re-matches (re-pattern (str %)) (str (:defined-by definition))) excluded-defined-by-syms-regex)
        (:export definition))))

(defn ^:private kondo-finding->diagnostic
  [{:keys [type message level row col end-row] :as finding}]
  (let [expression? (not= row end-row)
        finding (cond-> (merge {:end-row row :end-col col} finding)
                  expression? (assoc :end-row row :end-col col))]
    {:range (shared/->range finding)
     :tags (cond-> []
             (diagnostic-types-of-unnecessary-type type) (conj 1)
             (deprecated-diagnostic-types type) (conj 2))
     :message message
     :code (if-let [n (namespace type)]
             (str n "/" (name type))
             (name type))
     :severity (case level
                 :error   1
                 :warning 2
                 :info    3)
     :source (if (identical? :clojure-lsp/unused-public-var type)
               "clojure-lsp"
               "clj-kondo")}))

(defn ^:private valid-finding? [{:keys [row col level] :as finding}]
  (when (not= level :off)
    (or (and row col)
        (logger/warn "Invalid clj-kondo finding. Cannot find position data for" finding))))

(defn ^:private exclude-ns? [filename linter db]
  (when-let [namespace (shared/filename->namespace filename db)]
    (when-let [ns-exclude-regex-str (settings/get db [:linters linter :ns-exclude-regex])]
      (re-matches (re-pattern ns-exclude-regex-str) (str namespace)))))

(defn ^:private kondo-findings->diagnostics [filename linter db]
  (when-not (exclude-ns? filename linter db)
    (->> (get (:findings db) filename)
         (filter #(= filename (:filename %)))
         (filter valid-finding?)
         (mapv kondo-finding->diagnostic))))

(defn severity->level [severity]
  (case (int severity)
    1 :error
    2 :warning
    3 :info))

(defn severity->color [severity]
  (case (int severity)
    1 :red
    2 :yellow
    3 :cyan))

(defn ^:private clj-depend-violations->diagnostics [filename level db]
  (when-let [namespace (shared/filename->namespace filename db)]
    (mapv (fn [{:keys [message]}]
            (let [ns-definition (q/find-namespace-definition-by-filename db filename)]
              {:range (shared/->range ns-definition)
               :tags []
               :message message
               :code "clj-depend"
               :severity (case level
                           :error   1
                           :warning 2
                           :info    3)
               :source "clj-depend"}))
          (get-in db [:clj-depend-violations (symbol namespace)]))))

(defn find-diagnostics [^String uri db]
  (let [filename (shared/uri->filename uri)
        kondo-level (settings/get db [:linters :clj-kondo :level])
        depend-level (settings/get db [:linters :clj-depend :level] :info)]
    (if (shared/jar-file? filename)
      []
      (cond-> []
        (not= :off kondo-level)
        (concat (kondo-findings->diagnostics filename :clj-kondo db))

        (not= :off depend-level)
        (concat (clj-depend-violations->diagnostics filename depend-level db))))))

(defn ^:private publish-diagnostic!* [{:keys [diagnostics-chan]} diagnostic]
  (async/put! diagnostics-chan diagnostic))

(defn ^:private publish-all-diagnostics!* [{:keys [diagnostics-chan]} diagnostics]
  (async/onto-chan! diagnostics-chan diagnostics false))

(defn publish-diagnostics! [uri {:keys [db*], :as components}]
  (publish-diagnostic!* components {:uri uri
                                    :diagnostics (find-diagnostics uri @db*)}))

(defn publish-all-diagnostics! [paths {:keys [db*], :as components}]
  (let [db @db*]
    (publish-all-diagnostics!*
      components
      (eduction (map io/file)
                (mapcat file-seq)
                (map #(.getAbsolutePath ^java.io.File %))
                (map #(shared/filename->uri % db))
                (remove #(= :unknown (shared/uri->file-type %)))
                (map (fn [uri]
                       {:uri uri
                        :diagnostics (find-diagnostics uri db)}))
                paths))))

(defn publish-empty-diagnostics! [uri components]
  (publish-diagnostic!* components {:uri uri
                                    :diagnostics []}))

(defn ^:private unused-public-vars [var-defs kw-defs project-db kondo-config]
  (let [var-definitions (remove (partial exclude-public-diagnostic-definition? kondo-config) var-defs)
        var-nses (set (map :ns var-definitions)) ;; optimization to limit usages to internal namespaces, or in the case of a single file, to its namespaces
        var-usages (into #{}
                         (comp
                           (q/xf-all-var-usages-to-namespaces var-nses)
                           (map q/var-usage-signature))
                         (q/nses-and-dependents-analysis project-db var-nses))
        var-used? (fn [var-def]
                    (some var-usages (q/var-definition-signatures var-def)))
        kw-definitions (remove (partial exclude-public-diagnostic-definition? kondo-config) kw-defs)
        kw-usages (if (seq kw-definitions) ;; avoid looking up thousands of keyword usages if these files don't define any keywords
                    (into #{}
                          (comp
                            q/xf-all-keyword-usages
                            (map q/kw-signature))
                          (:analysis project-db))
                    #{})
        kw-used? (fn [kw-def]
                   (contains? kw-usages (q/kw-signature kw-def)))]
    (->> (concat (remove var-used? var-definitions)
                 (remove kw-used? kw-definitions))
         (map (fn [unused-var]
                (unused-public-var->finding unused-var kondo-config))))))

(defn ^:private file-var-definitions [project-db filename]
  (q/find-var-definitions project-db filename false))
(def ^:private file-kw-definitions q/find-keyword-definitions)
(def ^:private all-var-definitions q/find-all-var-definitions)
(def ^:private all-kw-definitions q/find-all-keyword-definitions)

(defn project-findings
  [db kondo-config]
  (let [project-db (q/db-with-internal-analysis db)]
    (unused-public-vars (all-var-definitions project-db)
                        (all-kw-definitions project-db)
                        project-db kondo-config)))

(defn files-findings
  [filenames db kondo-config]
  (let [project-db (q/db-with-internal-analysis db)
        files-db (update project-db :analysis select-keys filenames)]
    (unused-public-vars (all-var-definitions files-db)
                        (all-kw-definitions files-db)
                        project-db kondo-config)))

(defn file-findings
  [filename db kondo-config]
  (let [project-db (q/db-with-internal-analysis db)]
    (unused-public-vars (file-var-definitions project-db filename)
                        (file-kw-definitions project-db filename)
                        project-db kondo-config)))

(defn ^:private finalize-findings! [findings reg-finding!]
  (run! reg-finding! findings)
  (group-by :filename findings))

(defn custom-lint-project!
  [db {:keys [reg-finding! config]}]
  (-> (project-findings db config)
      (finalize-findings! reg-finding!)))

(defn custom-lint-files!
  [filenames db {:keys [reg-finding! config]}]
  (-> (files-findings filenames db config)
      (finalize-findings! reg-finding!)))

(defn custom-lint-file!
  [filename db {:keys [reg-finding! config]}]
  (-> (file-findings filename db config)
      (finalize-findings! reg-finding!)))
