(ns clj-kondo.hooks-api
  {:no-doc true}
  (:require
   [clj-kondo.impl.cache :as cache]
   [clj-kondo.impl.findings :as findings]
   [clj-kondo.impl.metadata :as meta]
   [clj-kondo.impl.namespace :as namespace]
   [clj-kondo.impl.rewrite-clj.node :as node]
   [clj-kondo.impl.rewrite-clj.node.protocols]
   [clj-kondo.impl.rewrite-clj.parser :as parser]
   [clj-kondo.impl.utils :as utils]
   [clojure.pprint]
   [sci.core :as sci])
  (:refer-clojure :exclude [macroexpand resolve]))

(defn- mark-generate [node]
  (assoc node :clj-kondo.impl/generated true))

(defn generated-node?
  "Checks whether node was generated by other hook, or if the node was
  written by user. Assumes that a node without location metadata was
  generated."
  [node]
  (or (:clj-kondo.impl/generated node)
      (let [m (meta node)]
        (or (:clj-kondo.impl/generated m)
            (not (:row m))))))

(defn parse-string [s]
  (parser/parse-string s))

(defn node? [n]
  (instance? clj_kondo.impl.rewrite_clj.node.protocols.Node n))

(defn keyword-node? [n]
  (utils/keyword-node? n))

(def keyword-node
  (comp mark-generate utils/keyword-node))

(defn string-node? [n]
  (instance? clj_kondo.impl.rewrite_clj.node.string.StringNode n))

(def string-node
  (comp mark-generate utils/string-node))

(defn token-node? [n]
  (instance? clj_kondo.impl.rewrite_clj.node.token.TokenNode n))

(def token-node
  (comp mark-generate utils/token-node))

(defn vector-node? [n]
  (and (instance? clj_kondo.impl.rewrite_clj.node.seq.SeqNode n)
       (identical? :vector (utils/tag n))))

(def ^:dynamic *reload* false)
(def ^:dynamic ^:private *debug* false)

(defn assert-children-nodes [children]
  (when *debug*
    (when-let [node (some #(when-not (node? %)
                             %) children)]
      (throw (new IllegalArgumentException (str "Not a node: " (str node)))))))

(defn vector-node [children]
  (assert-children-nodes children)
  (mark-generate (utils/vector-node children)))

(def list-node? utils/list-node?)

(defn list-node [children]
  (assert-children-nodes children)
  (mark-generate (utils/list-node children)))

(def set-node? utils/set-node?)

(defn set-node [children]
  (assert-children-nodes children)
  (mark-generate (utils/set-node children)))

(defn map-node? [n]
  (and (instance? clj_kondo.impl.rewrite_clj.node.seq.SeqNode n)
       (identical? :map (utils/tag n))))

(defn map-node [children]
  (assert-children-nodes children)
  (mark-generate (utils/map-node children)))

(defn sexpr [expr]
  (node/sexpr expr))

(defn tag [n]
  (node/tag n))

(defn reg-finding! [m]
  (let [ctx utils/*ctx*
        filename (:filename ctx)]
    (findings/reg-finding! ctx (assoc m :filename filename))))

(defn reg-keyword!
  [k reg-by]
  (utils/assoc-some k :reg reg-by))

(defn coerce [s-expr]
  (node/coerce s-expr))

(defn- var-definitions
  "Project cached analysis as a subset of public var-definitions."
  [analysis]
  (let [selected-keys [:ns :name
                       :fixed-arities :varargs-min-arity
                       :private :macro]]
    (->> (dissoc analysis :filename :source)
         (utils/map-vals #(if (map? %)
                            (select-keys % selected-keys)
                            %)))))

(defn- ns-analysis*
  "Adapt from-cache-1 to provide a uniform return format.
  Unifies the format of cached information provided for each source
  language."
  [lang ns-sym]
  (if (= lang :cljc)
    (->> (dissoc
          (cache/from-cache-1 (:cache-dir utils/*ctx*) :cljc ns-sym)
          :filename
          :source)
         (utils/map-vals var-definitions))
    (some->> (cache/from-cache-1 (:cache-dir utils/*ctx*) lang ns-sym)
             var-definitions
             (hash-map lang))))

(defn ns-analysis
  "Return any cached analysis for the namespace identified by ns-sym.
  Returns a map keyed by language keyword with values being maps of var
  definitions keyed by defined symbol. The value for each symbol is a
  subset of the values provide by the top level :analysis option."
  ([ns-sym] (ns-analysis ns-sym {}))
  ([ns-sym {:keys [lang]}]
   (if lang
     (ns-analysis* lang ns-sym)
     (reduce
      merge
      {}
      (map #(ns-analysis* % ns-sym) [:cljc :clj :cljs])))))

(defn walk
  [inner outer form]
  (cond
    (instance? clj_kondo.impl.rewrite_clj.node.protocols.Node form)
    (outer (update form :children #(mapv inner %)))
    :else (outer form)))

(defn prewalk
  [f form]
  (walk (partial prewalk f) identity (f form)))

(defn annotate
  {:no-doc true}
  [node original-meta]
  (let [!!last-meta (volatile! (assoc original-meta :derived-location true))]
    (prewalk (fn [node]
               (cond
                 (and (instance? clj_kondo.impl.rewrite_clj.node.seq.SeqNode node)
                      (identical? :list (utils/tag node)))
                 (if-let [m (meta node)]
                   (if-let [m (not-empty (select-keys m [:row :end-row :col :end-col]))]
                     (do (vreset! !!last-meta (assoc m :derived-location true))
                         (mark-generate node))
                     (-> (with-meta node
                           (merge @!!last-meta (meta node)))
                         mark-generate))
                   (->
                    (with-meta node
                      @!!last-meta)
                    mark-generate))
                 (instance? clj_kondo.impl.rewrite_clj.node.protocols.Node node)
                 (let [m (meta node)]
                   (if (:row m)
                     node
                     (-> (with-meta node
                           (merge @!!last-meta m))
                         mark-generate)))
                 :else node))
             node)))

(defn macroexpand [macro node bindings]
  (let [call (sexpr node)
        args (rest call)
        res (apply macro call bindings args)
        coerced (coerce res)
        annotated (annotate coerced (meta node))
        lifted (meta/lift-meta-content2 utils/*ctx* annotated)]
    ;;
    lifted))

(defn pprint [& args]
  (binding [*out* @sci/out]
    (apply clojure.pprint/pprint args)))

(defn resolve [{:keys [name call]}]
  (let [ctx utils/*ctx*
        ret (namespace/resolve-name ctx call (-> ctx :ns :name) name nil)]
    (select-keys ret [:ns :name])))

;; ctx call? ns-name name-sym expr
