(ns optimusbuf.parse
  (:require #?(:cljs [optimusbuf.util.slurp :include-macros true :refer [slurp]])
            #?(:cljs [cljs.pprint :refer [pprint]])
            #?(:clj [clojure.pprint :refer [pprint]])
            [optimusbuf.parse-textformat :refer [merge-tuple-into-map
                                                 xform-tf]]
            [optimusbuf.ast-transform :refer [recordify unnest]]
            [clojure.string :refer [join]]
            [instaparse.core :as insta :refer [parser]]))

(defn- make-msg-field [label type name fnum opts]
  [:field          label type name fnum opts])

(defn- make-map-field [ktype vtype name fnum opts]
  [:mapField       ktype vtype name fnum opts])

(defn- make-oneof-field [type name fnum opts]
  [:oneofField       type name fnum opts])

(defn- make-enum-field [name fnum opts]
  [:enumField       name fnum opts])

(defn- make-rpc [name input output opts]
  [:rpc      name input output opts])

(defn- make-message [name body]
  (into [:message name] body))

(defn- make-enum [name body]
  (into [:enum name] body))

(defn- make-group [label name fnum body]
  (into [:group label name fnum] body))

(defn- make-service [name body]
  (into [:service name] body))


(defn- xform-opts
  "xform ... of [:opts ...] where ... is like
     [[:a 1]
      [:a 2]
      [:b 1]
      [:b 2]
      [:a [3 4 5]]]
   into
     [:opts {:a [1 2 3 4 5]
             :b [1 2]}]"
  [opts]
  [:opts (reduce merge-tuple-into-map {} opts)])

(defn- merge-opt-ext-res
  "[[:option 'n1' 'v1']
    [:option 'n2' 'v2']
    [:reserved-names ['name1'] ['name2']]
    [:reserved-names ['name3'] ['name4']]
    [:reserved-ranges [1 1] [5 10]]
    [:reserved-ranges [100 1000] [2000 max]]
    [:extensions [1001 1100] [1201 1300]]
    [:extensions [1401 1500] [1601 1600]]
    [:field ...]
    [:field ...]]
   ==>
   [[:field ...]
    [:field ...]
    [:exts [1001 1100] [1201 1300] [1401 1500] [1601 1600]]
    [:rnames ['name1' 'name2' 'name3' 'name4']]
    [:rranges [1 1] [5 10] [100 1000] [2000 max]]
    [:opts [['n1' 'v1'] ['n2' 'v2']]]]"
  [body]
  (let [opts (transient [:opts])
        exts (transient [:exts])
        rnames (transient [:rnames])
        rranges (transient [:rranges])
        xformer #(case (first %)
                   :option (do (conj! opts (rest %)) nil)
                   :extensions (do (reduce conj! exts (rest %)) nil)
                   :reserved-names (do (reduce conj! rnames (second %)) nil)
                   :reserved-ranges (do (reduce conj! rranges (rest %)) nil)
                   %)
        xbody (->> (map xformer body)
                   (into [])
                   (doall))
        gathered (reduce conj xbody [(persistent! exts)
                                     (persistent! rnames)
                                     (persistent! rranges)
                                     (-> (persistent! opts)
                                         rest
                                         xform-opts)])]
    (->> gathered
         (remove nil?)
         (into []))))

(defn- merge-option [body]
  (let [tags (transient [:opts])
        xformer #(if (= (first %) :option)
                   (do (conj! tags (into [] (rest %))) nil)
                   %)
        xbody (->> (map xformer body)
                   (into [])
                   (remove nil?)
                   (doall))
        tags (-> (persistent! tags)
                 rest
                 xform-opts)]
    (conj (into [] xbody) tags)))

(defn- xform-label
  ([] [:label nil])
  ([form] [:label (keyword form)]))

(defn- xform-message-field
  "xform ... of [:field ...] where ...
     [:label 'required'] ; [:label] implies 'optional'
     [:type :string]
     [:fieldName 'field1']
     [:fieldNumber 1]]
     [:options [:deprecated true]
               [:whatever   false]]"
  ([a1 a2 a3 a4]
   (xform-message-field a1 a2 a3 a4 nil))
  ([[_ a1] [_ a2] [_ a3] [_ a4] a5]
   (make-msg-field a1 a2 a3 a4 a5)))

(defn- xform-enum-field
  "xform [:enumField \"TWO\" 2 [:opts ...]]"
  ([fname num]
   (xform-enum-field fname num [:opts]))
  ([fname num opts]
   (make-enum-field fname num opts)))

(defn- xform-mapField
  "xform ... of [:mapField ...] where ...
    [:keyType :uint32]
    [:valType [:priType 'string']]
    [:mapName 'kv1']
    [:fieldNumber 4]
    [:options [:deprecated true]
              [:whatever   false]]"
  ([a1 a2 a3 a4] (xform-mapField a1 a2 a3 a4 nil))
  ([a1 [_ a2] [_ a3] [_ a4] a5]
   (make-map-field a1 a2 a3 a4 a5)))

(defn- xform-ranges
  "xform
    [[:range 10] [:range 20 30] [:range 40 \"max\"]]"
  [ranges]
  (->> ranges
       (map #(with-meta (vector (second %) (last %)) (meta %)))
       (clojure.walk/prewalk-replace {"max" 536870911}) ; max field number 2^29-1
       (vec)))

(defn- xform-names
  "xform ... of [:reserved-names ...] where ...
    [:fieldName :xyz] [:fieldName :abc]"
  [& names]
  (->> names
       (map #(-> % (second) (vector) (with-meta (meta %))))
       (into [:reserved-names])))

(defn- xform-oneofField
  "xform ... of [:oneofField ...] where ...
   [:type :string]
   [:fieldName :one_str]
   [:fieldNumber 6]
   [:options [:deprecated true]
             [:whatever   false]]"
  ([a1 a2 a3] (xform-oneofField a1 a2 a3 nil))
  ([[_ a1] [_ a2] [_ a3] a4] (make-oneof-field a1 a2 a3 a4)))

(defn- xform-rpc
  "xform ... of [:rpc ...] where ...
    :Rpc2 :Msg1 :Msg2
    [:option :java_compiler \"javac\"]
    [:option :whatever false]"
  ([[_ a1] a2 a3 & opts]
   (make-rpc a1 a2 a3 (xform-opts (map rest opts)))))

(defn- xform-group
  "xfrom ... of [:group ...] where ... =
    [:label \"repeated\"]
    \"Result\"
    [:fieldNumber 1]
    [:field \"url\" 2 :string :required nil]
    [:field \"title\" 3 :string :optional nil]
    [:field \"snippets\" 4 :string :repeated nil]] ..."
  [[_ a1] a2 [_ a3] & args]
  (make-group a1 a2 a3 (merge-opt-ext-res args)))

(defn- xform-signedFloatLit
  [& args] (let [s (join "" args)]
             (case s
               "inf" ##Inf
               "nan" ##NaN
               (parse-double s))))

(defn- parseInt [val base]
  #?(:clj (Integer/parseInt val base))
  #?(:cljs (js/parseInt val base)))

(def ^:private xform
  (merge
   xform-tf
   {:proto #(merge-option %&)
    ; names
    :ident identity
    :messageName identity
    :enumName identity
    :groupName identity
    :serviceName identity
    :oneofName identity
    ; syntax type
    :label xform-label
    :fullIdent str
    :keyType keyword
    :priType keyword
    :messageType (fn [& args] (join "" args))
    :returnType identity
    ; int literals
    :decimalLit read-string
    :octalLit read-string
    :hexLit read-string
    :intLit identity
    ; string literals
    :hexDigit2 identity
    :hexEscape #(char (parseInt % 16))
    :octalDigit3 identity
    :octEscape #(char (parseInt % 8))
    :charValue identity
    :strLit str
    ; other literals
    :boolLit parse-boolean
    :signedFloatLit xform-signedFloatLit
    ; top(ish) levels xforms
    :rpc xform-rpc
    :group xform-group
    :enum #(make-enum %1 (merge-option %&))
    :message #(make-message %1 (merge-opt-ext-res %&))
    :service #(make-service %1 (merge-option %&))
    :extensions (fn [& args] (into [:extensions] (xform-ranges args)))
    :reserved-ranges (fn [& args] (into [:reserved-ranges] (xform-ranges args)))
    :reserved-names xform-names
    ; field
    :field xform-message-field
    :mapField xform-mapField
    :oneofField xform-oneofField
    :enumField xform-enum-field
    ; option
    :fieldOption #(into [] %&)
    :fieldOptions #(xform-opts (into [] %&))
    :optionName str
    ; textformat -- merged from xform-tf
    }))

(def ^:private void
  "Matches whitespaces and comments; to be used as :auto-whitespace param when
   creating instaparse parser."
  (insta/parser
   "void = { #'\\s+' | #'(\\/\\*)[\\s\\S]*?(\\*\\/)' | '//' #'.*' }"))

(def ^:private parser-2
  (parser (str (slurp "resources/ebnf/proto2.ebnf")
               (slurp "resources/ebnf/textformat.ebnf"))
          :auto-whitespace void))

(def ^:private parser-3
  (parser (str (slurp "resources/ebnf/proto3.ebnf")
               (slurp "resources/ebnf/textformat.ebnf"))
          :auto-whitespace void))

(def ^:private parser-ver
  (parser (slurp "resources/ebnf/protover.ebnf")
          :auto-whitespace void))

(defn- parse2 [text]
  (let [ast (parser-2 text)
        ast+line (insta/add-line-and-column-info-to-metadata text ast)]
    (insta/transform xform ast+line)))

(defn- parse3 [text]
  (let [ast (parser-3 text)
        ast+line (insta/add-line-and-column-info-to-metadata text ast)]
    (insta/transform xform ast+line)))

(defn parse [text]
  (if (= [:syntax [:version "proto3"]] (parser-ver text))
    (parse3 text)
    (parse2 text)))
