(ns com.github.ivarref.paginate-vector
  (:require [com.github.ivarref.paginate-vector.impl.multi-last :as multi-last]
            [com.github.ivarref.paginate-vector.impl.multi-first :as multi-first]
            [clojure.edn :as edn]))


(defn get-cursor [opts]
  (or
    (let [cursor (or (get opts :after) (get opts :before))]
      (when (and (string? cursor) (not-empty cursor))
        (edn/read-string cursor)))
    {}))


(defn get-context [cursor]
  (when (and (string? cursor) (not-empty cursor))
    (:context (edn/read-string cursor))))


(defn paginate
  ([vectors f opts]
   (paginate vectors f opts nil))
  ([vectors f opts context]
   (assert (fn? f) "Expected f to be a function")
   (assert (vector? vectors) "Expected vectors to be a vector")
   (assert (every? vector? vectors) "Expected every item in vectors to be a vector")
   (assert (map? opts) "Expected opts to be a map")
   (let [cursor (get-cursor opts)
         context (or (:context cursor) context)
         cursor (-> cursor
                    (assoc :context context)
                    (update :counts (fn [old-val] (or old-val (mapv count vectors)))))
         num-vecs (count (get cursor :counts))]
     (when (= num-vecs (count vectors))
       (doseq [[idx cnt] (map-indexed vector (:counts cursor))]
         (when (> cnt (count (nth vectors idx)))
           (throw (ex-info "Vector cannot shrink" {:vector (nth vectors idx)
                                                   :old-count cnt})))))
     (cond
       (not= num-vecs (count vectors))
       (throw (ex-info "Number of vectors cannot grow or shrink" {:opts opts}))

       (and (some? (get opts :first)) (some? (get opts :last)))
       (throw (ex-info "Both :first and :last given, don't know what to do." {:opts opts}))

       (and (some? (get opts :first)) (some? (get opts :before)))
       (throw (ex-info ":first and :before given, please use :first with :after" {:opts opts}))

       (and (some? (get opts :last)) (some? (get opts :after)))
       (throw (ex-info ":last and :after given, please use :last with :before" {:opts opts}))

       (empty? vectors)
       {:edges    []
        :pageInfo {:hasNextPage false
                   :hasPrevPage false
                   :startCursor (pr-str {:context context})
                   :endCursor   (pr-str {:context context})
                   :totalCount  0}}

       (pos-int? (get opts :first))
       (multi-first/paginate-multi-first vectors f (get opts :first) cursor)

       (pos-int? (get opts :last))
       (multi-last/paginate-multi-last (vec (reverse vectors)) f (get opts :last) cursor)

       :else
       (throw (ex-info "Bad opts given, expected either :first or :last to be a positive integer." {:opts opts}))))))
