(ns com.github.ivarref.paginate-vector
  (:require [com.github.ivarref.paginate-vector.impl.bst-multi :as bm]
            [com.github.ivarref.paginate-vector.impl.bst-multi-last :as bml]
            [clojure.edn :as edn]))


(defn- get-cursor [opts]
  (or
    (let [cursor (or (get opts :after) (get opts :before))]
      (when (and (string? cursor) (not-empty cursor))
        (edn/read-string cursor)))
    {}))


(defn get-context [opts]
  (:context (get-cursor opts)))


(defn prepare-paginate
  [opts inp-vector]
  (bm/balanced-tree-multi inp-vector opts))


(defn paginate
  [prepared f keep? opts context]
  (assert (fn? f) "Expected f to be a function")
  (assert (fn? keep?) "Expected keep? to be a function")
  (assert (map? opts) "Expected opts to be a map")
  (let [cursor-str (or (get opts :after) (get opts :before))]
    (cond
      (and (some? (get opts :first)) (some? (get opts :last)))
      (throw (ex-info "Both :first and :last given, don't know what to do." {:opts opts}))

      (and (some? (get opts :first)) (some? (get opts :before)))
      (throw (ex-info ":first and :before given, please use :first with :after" {:opts opts}))

      (and (some? (get opts :last)) (some? (get opts :after)))
      (throw (ex-info ":last and :after given, please use :last with :before" {:opts opts}))

      (nil? (:root prepared))
      {:edges    []
       :pageInfo {:hasNextPage false
                  :hasPrevPage false
                  :startCursor (pr-str {:context context})
                  :endCursor   (pr-str {:context context})
                  :totalCount  0}}

      (pos-int? (get opts :first))
      (bm/paginate-first prepared
                         {:max-items (get opts :first)
                          :f         f
                          :keep?     keep?
                          :context   context}
                         cursor-str)

      (pos-int? (get opts :last))
      (bml/paginate-last prepared
                         {:max-items (get opts :last)
                          :f         f
                          :keep?     keep?
                          :context   context}
                         cursor-str)

      :else
      (throw (ex-info "Bad opts given, expected either :first or :last to be a positive integer." {:opts opts})))))
