(ns norled.common.serialization
  (:gen-class)
  (:require [taoensso.nippy :as nippy]
            [cheshire.core :as json])
  (:import [org.apache.kafka.common.serialization Deserializer Serde Serializer]
           [java.time Instant]
           org.apache.kafka.streams.processor.TimestampExtractor))

(defmulti serialize-key (fn [topic data] topic))
(defmulti serialize-value (fn [topic data] topic))
(defmulti deserialize-key (fn [topic data] topic))
(defmulti deserialize-value (fn [topic data] topic))

(defmethod serialize-key :default [_ data]
  (.getBytes data "UTF-8"))

(defmethod deserialize-key :default [_ data]
  (String. data "UTF-8"))

(defmethod serialize-value :default [_ data] (nippy/freeze data))
(defmethod deserialize-value :default [_ data] (nippy/thaw data))

(deftype DelegatedKeySerializer []
  Serializer
  (configure [_ _ _])
  (close [_])
  (serialize [_ topic data]
    (some->> data (serialize-key topic)))

  Deserializer
  (deserialize [_ topic data]
    (some->> data (deserialize-key topic))))

(deftype DelegatedValueSerializer []
  Serializer
  (configure [_ _ _])
  (close [_])
  (serialize [_ topic data]
    (some->> data (serialize-value topic)))

  Deserializer
  (deserialize [_ topic data]
    (some->> data (deserialize-value topic))))

(deftype DelegatedSerde [^:volatile-mutable is-key?]
  Serde
  (configure [_ _ is-key]
    (set! is-key? is-key))
  (close [_])
  (serializer [this]
    (if is-key?
      (DelegatedKeySerializer.)
      (DelegatedValueSerializer.)))
  (deserializer [this]
    (if is-key?
      (DelegatedKeySerializer.)
      (DelegatedValueSerializer.))))

(defn delegated-key-serde [] (->DelegatedSerde true))
(defn delegated-value-serde [] (->DelegatedSerde false))

(defn delegated-key-serializer-class-name [] (pr-str DelegatedKeySerializer))
(defn delegated-value-serializer-class-name [] (pr-str DelegatedValueSerializer))

(deftype InstFieldTimestampExtractor []
  TimestampExtractor
  (extract [_ record _]
    (-> record .value :inst .getTime)))

(deftype JsonTimestampExtractor []
  TimestampExtractor
  (extract [_ record _]
    (let [json-data (json/parse-string (:value record) true)
          iso-timestamp (:timestamp json-data)
          instant (java.time.Instant/parse iso-timestamp)]
      (.toEpochMilli instant))))

(defn assert-serializable [m key-order]
  (when-not (map? m)
    (throw (ex-info "Expected value to be a map"
                    {:value m
                     :key-order key-order})))
  (let [expected-keys (conj (set key-order) :inst)
        actual-keys (set (keys m))]
    (when-not (= actual-keys expected-keys)
      (throw (ex-info (str "Mismatch between expected and actual keys in map"
                           (pr-str {:expected-keys expected-keys
                                    :actual-keys actual-keys
                                    :map m}))
                      {:expected-keys expected-keys
                       :actual-keys actual-keys
                       :map m})))))

(defn map->tuple [m key-order]
  (assert-serializable m key-order)
  [(.getTime (:inst m)) (mapv m key-order)])

(defn assert-deserializable [[epoch-ms values :as tuple] key-order]
  (when-not (sequential? values)
    (throw (ex-info "Expected values to be sequential" {:values values :type (type values)})))
  (when-not (integer? epoch-ms)
    (throw (ex-info "Expected to find timestamp in tuple" {:tuple tuple}))))

(defn tuple->map [[epoch-ms values :as tuple] key-order]
  (assert-deserializable tuple key-order)
  (-> (zipmap key-order values)
      (assoc :inst (java.util.Date. epoch-ms))))

(defn serialize [m key-order]
  (nippy/freeze (map->tuple m key-order)))

(defn deserialize [nippy-buf key-order]
  (tuple->map (nippy/thaw nippy-buf) key-order))

(deftype NippySerializer []
  Serializer
  (configure [_ _ _])
  (serialize [_ _ data]
    (nippy/freeze data))
  (close [_]))

(deftype NippyDeserializer []
  Deserializer
  (configure [_ _ _])
  (deserialize [x y data]
    (some-> data nippy/thaw))
  (close [_]))

(deftype NippySerde []
  Serde
  (deserializer [_]
    (NippyDeserializer.))
  (serializer [_]
    (NippySerializer.))
  (configure [_ _ _])
  (close [_]))
