(ns floj.wave-lexicon
  (:require [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [floj.calibration :as calibrate]
            [floj.io :as fio]
            [floj.record :as record]
            [floj.state :as state]
            [floj.stream-manager :as stream]
            [floj.brainflow.board-shim :as brainflow]
            [floj.categorization :as category]
            [floj.api :as api]))

(def ^:private MIN_SAMPLES_FOR_SIGNATURE 20)
(def ^:private SIMILARITY_THRESHOLD 0.75)
(def DEFAULT_RECORDING_DURATION 5000)

(defn fill-initial-lexicon!
  "Initialize the wave lexicon directory structure and create default categories"
  []
  (let [default-profile-name (:name ((:get-active-profile @state/state)))]
    (try
      (let [pong-dir (fio/get-wave-lexicon-dir default-profile-name "pong")
            pong-up-dir (str pong-dir "/up")
            pong-down-dir (str pong-dir "/down")]

        (fio/ensure-directory! pong-dir)
        (fio/ensure-directory! pong-up-dir)
        (fio/ensure-directory! pong-down-dir)

        (spit (str pong-dir "/README.md")
              "# Pong Wave Signatures\n\nThis directory contains your recorded brain patterns for controlling the Pong game.\n\n- The 'up' directory contains recordings when you're thinking about moving the paddle up\n- The 'down' directory contains recordings when you're thinking about moving the paddle down\n\nTo add new recordings, use the 'A' key in the CLI and select the appropriate category.")

        (spit (str pong-up-dir "/README.md")
              "# Up Movement Recordings\n\nThis directory will contain your recorded brain patterns for moving the paddle upward.\nTo add a new recording, use the 'A' key in the CLI.")

        (spit (str pong-down-dir "/README.md")
              "# Down Movement Recordings\n\nThis directory will contain your recorded brain patterns for moving the paddle downward.\nTo add a new recording, use the 'A' key in the CLI.")

        (let [config-path (str (fio/config-base-dir) "/config.edn")
              config (if (.exists (io/file config-path))
                       (edn/read-string (slurp config-path))
                       {})
              updated-config (update config :wave-lexicon-categories
                                     #(if (seq %)
                                        (if (some #{:pong-up :pong-down} %)
                                          %
                                          (conj % :pong-up :pong-down))
                                        [:pong-up :pong-down]))]
          (spit config-path (pr-str updated-config))))
      (println (str "Loaded " default-profile-name "'s wave-lexicon!"))
      (catch Exception e
        (println "Error initializing wave lexicon:" (.getMessage e))
        (.printStackTrace e)))))

(defn list-wave-signature-categories
  "List all wave signature categories for a profile"
  []
 (try
   (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
         lexicon-dir (fio/get-wave-lexicon-dir profile-name "")]
     (if (.exists (io/file lexicon-dir))
       (let [dirs (filter #(.isDirectory %)
                          (or (seq (.listFiles (io/file lexicon-dir))) []))
             categories (map #(.getName %) dirs)]
         (println "Found" (count categories) "lexicon categories:")
         (doseq [category categories]
           (println "  -" category))
         categories)
       (do
         (println "No lexicon directory found for profile" profile-name)
         [])))
   (catch Exception e
     (println "Error listing lexicon categories:" (.getMessage e))
     (.printStackTrace e)
     [])))

(defn list-wave-signatures-in-category
  "List all wave signatures within a category"
  [profile-name category]
  (try
    (let [category-dir (str (fio/config-base-dir) "/profiles/" profile-name "/wave_lexicon/" category)]
      (when (.exists (io/file category-dir))
        (->> (.listFiles (io/file category-dir))
             (filter #(.isDirectory %))
             (map #(.getName %)))))
    (catch Exception e
      (println (str "Error listing wave signatures in category " category ": " (.getMessage e)))
      [])))

(defn list-signature-recordings
  "List all recordings for a specific signature"
  [profile-name category signature]
  (try
    (let [signature-dir (str (fio/config-base-dir) "/profiles/" profile-name "/wave_lexicon/"
                             category "/" signature)]
      (when (.exists (io/file signature-dir))
        (->> (.listFiles (io/file signature-dir))
             (filter #(.isDirectory %))
             (map #(.getName %))
             (sort))))
    (catch Exception e
      (println (str "Error listing recordings for signature " signature ": " (.getMessage e)))
      [])))

(defn list-only-signatures
  "List all signatures in a category"
  [profile-name category]
  (try
    (let [category-dir (fio/get-wave-lexicon-dir profile-name category)]
      (when (.exists (io/file category-dir))
        (->> (io/file category-dir)
             .listFiles
             (filter #(.isDirectory %))
             (map #(.getName %))
             sort
             vec)))
    (catch Exception e
      (println "Error listing wave signatures:" (.getMessage e))
      [])))

(defn list-all-wave-signatures
  "List all categories and their signatures in a user's wave lexicon"
  []
  (try
    (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
          lexicon-dir (fio/get-wave-lexicon-dir profile-name "")]
      (if (.exists (io/file lexicon-dir))
        (let [categories (list-wave-signature-categories)]
          (if (seq categories)

            (doseq [category categories]
              (println "\tCategory:" category)
              (let [signatures (list-wave-signatures-in-category profile-name category)]
                (if (seq signatures)
                  (doseq [signature signatures]
                    (let [recordings (list-signature-recordings profile-name category signature)
                          recording-count (count recordings)]
                      (println "\t  - Signature:" signature
                               (str "(" recording-count " recordings)"))))
                  (println "  No signatures in this category"))))
            (println "No categories found in wave lexicon"))
          categories)
        (do
          (println "No lexicon directory found for profile" profile-name)
          [])))
    (catch Exception e
      (println "Error listing lexicon categories:" (.getMessage e))
      (.printStackTrace e)
      [])))

(defn list-wave-signatures-by-category
  "List all wave signatures for a specific category"
  [_]
  (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
        categories (list-wave-signature-categories)]

    (if (seq categories)
      (do
        (println "Available categories:")
        (doall (map-indexed #(println (str (inc %1) ". " %2)) categories))

        (print "\nSelect category number: ")
        (flush)
        (let [input (read-line)
              idx (try (Integer/parseInt input) (catch Exception _ -1))]

          (when (and (>= idx 1) (<= idx (count categories)))
            (let [category (nth categories (dec idx))
                  dir (fio/get-wave-lexicon-dir profile-name category)
                  signatures (when (.exists (io/file dir))
                               (->> (.listFiles (io/file dir))
                                    (filter #(.isDirectory %))
                                    (sort-by #(.getName %))))]

              (if (seq signatures)
                (do
                  (println (format "\nWave signatures for category '%s':" category))
                  (doseq [sig signatures]
                    (let [metadata-file (io/file (str (.getPath sig) "/recording_metadata.edn"))
                          metadata (when (.exists metadata-file)
                                     (try
                                       (clojure.edn/read-string (slurp metadata-file))
                                       (catch Exception _ nil)))
                          timestamp (or (:recorded-at metadata) "Unknown")
                          sample-count (or (:sample-count metadata) "Unknown")]
                      (println (format "  %s (%s samples, recorded at %s)"
                                       (.getName sig)
                                       sample-count
                                       timestamp)))))
                (println (format "\nNo signatures found for category '%s'." category))))))
        (println))
      (println "\nNo wave signature categories found."))))

(defn save-wave-lexicon-entry!
  "Save a wave lexicon entry for a specific category"
  [profile-name category entry]
  (try
    (let [timestamp (System/currentTimeMillis)
          category-dir (fio/get-wave-lexicon-dir profile-name category)
          entry-path (str category-dir "/" category "_" timestamp)
          metadata-path (str entry-path "/recording_metadata.edn")]
      (fio/ensure-directory! category-dir)
      (fio/ensure-directory! entry-path)

      (spit metadata-path (pr-str (assoc entry
                                         :category category
                                         :created-at (java.util.Date.)
                                         :recording-id (str category "_" timestamp))))
      entry-path)
    (catch Exception e
      (println "Error saving wave lexicon entry:" (.getMessage e))
      (.printStackTrace e)
      nil)))

(defn compute-tensor-difference
  "Compute difference between current band powers and golden tensor"
  [band-powers golden-tensor]
  (let [keys-to-compare (keys golden-tensor)]
    (into {} (map (fn [k]
                    [k (- (get band-powers k 0) (get golden-tensor k 0))])
                  keys-to-compare))))

(defn match-brain-activity
  "Match current brain activity against lexicon entries"
  [current-data]
  (try
    (let [profile ((:get-active-profile @state/state))
          profile-name (:name profile)
          lexicon-dir (str ".home/.lor/profiles/lexicon/" profile-name)

          ; Process current data
          sampling-rate (brainflow/get-sampling-rate (brainflow/get-board-id @state/shim))
          current-powers (calibrate/extract-band-powers current-data sampling-rate)

          ; Load all lexicon entries
          entries (when (.exists (io/file lexicon-dir))
                    (for [file (.listFiles (io/file lexicon-dir))
                          :when (.isFile file)
                          :let [entry (try
                                        (edn/read-string (slurp file))
                                        (catch Exception e nil))]
                          :when entry]
                      entry))

          ; Calculate similarity scores
          scores (for [entry entries
                       :let [entry-powers (:band-powers entry)
                             ; Calculate Euclidean distance between power spectra
                             distance (reduce +
                                              (for [channel-idx (range (min (count current-powers)
                                                                            (count entry-powers)))]
                                                (let [current-ch (nth current-powers channel-idx {})
                                                      entry-ch (nth entry-powers channel-idx {})
                                                      keys (into #{} (concat (keys current-ch) (keys entry-ch)))
                                                      dist-sq (reduce +
                                                                      (for [k keys]
                                                                        (let [curr-val (get current-ch k 0.0)
                                                                              entry-val (get entry-ch k 0.0)
                                                                              diff (- curr-val entry-val)]
                                                                          (* diff diff))))]
                                                  dist-sq)))
                             similarity (/ 1.0 (+ 1.0 (Math/sqrt distance)))]]
                   {:label (:label entry)
                    :similarity similarity})

          ; Sort by similarity (highest first)
          sorted-scores (sort-by :similarity > scores)

          ; Get the best match
          best-match (first sorted-scores)]

      (if best-match
        (do
          (println "Best match:" (:label best-match) "with similarity" (:similarity best-match))
          best-match)
        (do
          (println "No lexicon entries to match against")
          nil)))
    (catch Exception e
      (println "Error matching brain activity:" (.getMessage e)))))

(defn create-wave-signature-context
  "Create a recording context for a wave signature"
  [profile-name category signature-name & {:keys [include-in-aggregation] :or {include-in-aggregation true}}]
  (try
    (let [timestamp (System/currentTimeMillis)
          signature-dir (str (fio/get-wave-lexicon-dir profile-name category)
                             "/" (str/lower-case signature-name) "_" timestamp)
          board-id (brainflow/get-board-id @state/shim)
          device-name (brainflow/get-device-name board-id)
          metadata {:recording-id (str "wave_signature_" timestamp)
                    :start-time timestamp
                    :recorded-at (java.util.Date.)
                    :device-type device-name
                    :board-id board-id
                    :sampling-rate api/CURRENT_SRATE
                    :category (name category)
                    :signature-name (name signature-name)
                    :is-wave-signature true
                    :include-in-aggregation include-in-aggregation
                    :version "1.0"
                    :channel-count (brainflow/get-eeg-channels board-id)}

          context {:lorfile-dir signature-dir
                   :metadata metadata}]
      context)
    (catch Exception e
      (println "Error creating wave signature context:" (.getMessage e))
      (.printStackTrace e)
      nil)))

(defn start-wave-signature-recording!
  "Start recording a wave signature"
  [category signature-name & {:keys [include-in-aggregation] :or {include-in-aggregation true}}]
  (try
    (when (or (str/blank? category) (str/blank? signature-name))
      (throw (Exception. "Category and signature name must be provided")))

    (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
          context (create-wave-signature-context profile-name category signature-name
                                                 :include-in-aggregation include-in-aggregation)
          recording-dir (str (fio/get-wave-lexicon-dir profile-name category)
                             "/" signature-name)
          recording-info {:recording-type signature-name
                          :path recording-dir}]

      (when-not context
        (throw (Exception. "Failed to create recording context")))

      ; Store context for recording
      (reset! state/recording-context context)

      ; Add tag for wave signature start
      (swap! state/tags conj {:timestamp (:start-time (:metadata context))
                              :label (str "WAVE_SIGNATURE_START:"
                                          (name category) "/" (name signature-name))})

      ; Reset EEG data collection
      (reset! state/eeg-data [])

      ; Start the actual recording
      (record/start-recording! recording-info)

      ; Log success
      (println "Started recording wave signature for" category "/" signature-name
               (if include-in-aggregation "" "(practice mode - will not be aggregated)"))

      ; Return context for reference
      context)
    (catch Exception e
      (println "Error starting wave signature recording:" (.getMessage e))
      (.printStackTrace e)
      nil)))

(defn create-category-context
  "Create a recording context for a wave signature category"
  [profile-name category]
  (try
    (let [timestamp (System/currentTimeMillis)
          category-dir (str (fio/get-wave-lexicon-dir profile-name category)
                            "/" (str/lower-case category) "_recordings/")
          board-id (brainflow/get-board-id @state/shim)
          device-name (brainflow/get-device-name board-id)
          metadata {:recording-id (str category "_" timestamp)
                    :start-time timestamp
                    :recorded-at (java.util.Date.)
                    :device-type device-name
                    :board-id board-id
                    :sampling-rate api/CURRENT_SRATE
                    :category-name (name category)
                    :is-category true
                    :version "1.0"
                    :signatures-count (brainflow/get-eeg-channels board-id)}

          context {:lorfile-dir category-dir
                   :metadata metadata}]
      context)
    (catch Exception e
      (println "Error creating wave signature context:" (.getMessage e))
      (.printStackTrace e)
      nil)))

(defn start-category-recording!
  "Start recording a whole category"
  [category]
  (try
    (when (str/blank? category)
      (throw (Exception. "Category and signature name must be provided")))

    (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
          _ (println "profile " profile-name)
          context (create-category-context profile-name category)
          category-recording-dir (:lorfile-dir context)
          _ (println "fixing the PATH!!! " category-recording-dir)
          category-recording-info {:recording-type category
                                   :path category-recording-dir}]

      (when-not context
        (throw (Exception. "Failed to create recording context")))

      ; Store context for recording
      (reset! state/recording-context context)
      (println "state: " @state/recording-context)
      ; Add tag for wave signature start
      (swap! state/tags conj {:timestamp (:start-time (:metadata context))
                              :label (str (str/upper-case category) "_START:"
                                          (name category))})

      ; Reset EEG data collection
      (reset! state/eeg-data [])

      ; Start the actual recording
      (record/start-recording! category-recording-info)

      ; Log success
      (println "Started recording wave signature for" category)
      context)
    (catch Exception e
      (println "Error starting category recording:" (.getMessage e))
      (.printStackTrace e)
      nil)))

(defn calculate-correlation
  "Calculate correlation between two signals"
  [signal1 signal2]
  (try
    (let [n (count signal1)
          mean1 (/ (reduce + signal1) n)
          mean2 (/ (reduce + signal2) n)

          variance1 (/ (reduce + (map #(Math/pow (- % mean1) 2) signal1)) n)
          variance2 (/ (reduce + (map #(Math/pow (- % mean2) 2) signal2)) n)

          std1 (Math/sqrt variance1)
          std2 (Math/sqrt variance2)

          covariance (/ (reduce +
                                (map #(* (- %1 mean1) (- %2 mean2))
                                     signal1 signal2))
                        n)
          correlation (if (and (> std1 0) (> std2 0))
                        (/ covariance (* std1 std2))
                        0.0)]
      correlation)
    (catch Exception e
      (println "Error calculating correlation:" (.getMessage e))
      0.0)))

(defn extract-signature-features
  "Extract features from EEG data for signature matching"
  [eeg-data sampling-rate]
  (try
    (println "Extracting features from" (count eeg-data) "data points")
    (when (empty? eeg-data)
      (throw (Exception. "No EEG data provided")))
    (let [data-format (cond
                        (and (seq eeg-data) (map? (first eeg-data))) :map-format
                        (and (vector? eeg-data) (every? vector? eeg-data)) :vector-of-vectors
                        :else :unknown)
          ;; Convert to proper format [channels][samples]
          _ (println "eeg-data " eeg-data)
          normalized-data (case data-format
                            :map-format
                            (let [channel-keys (remove #{:timestamp} (keys (first eeg-data)))]
                              (vec (for [k channel-keys]
                                     (mapv #(get % k 0.0) eeg-data))))
                            :vector-of-vectors
                            (apply mapv vector (map rest eeg-data))

                            (stream/normalize-data-format eeg-data))
          _ (when (or (not (vector? normalized-data))
                      (empty? normalized-data)
                      (not (vector? (first normalized-data))))
              (throw (Exception. (str "Invalid data format for band power extraction: "
                                      (type normalized-data)))))

          band-powers (calibrate/extract-band-powers normalized-data sampling-rate)

          ; EEG Data looks like: [{:eeg [[147.0 ch1 ch2 ch3 ch4] ...
          ; Convert to proper format [channels][samples]
          raw-eeg-matrix (-> eeg-data first :eeg) ; Pull out the actual matrix
          without-timestamps (mapv rest raw-eeg-matrix) ; Drop timestamps
          vector-data (apply mapv vector without-timestamps)

          ; Calculate temporal features (amplitude variation)
          amplitude-stats (for [channel vector-data]
                            (let [mean (/ (reduce + channel) (count channel))
                                  variance (/ (reduce + (map #(Math/pow (- % mean) 2) channel))
                                              (count channel))
                                  std-dev (Math/sqrt variance)
                                  min-val (apply min channel)
                                  max-val (apply max channel)
                                  range-val (- max-val min-val)]
                              {:mean mean
                               :std-dev std-dev
                               :range range-val
                               :min min-val
                               :max max-val}))
          _ (println "amplitude-stats" amplitude-stats)
          ; Calculate coherence between channels (simple version)
          channel-coherence (when (> (count vector-data) 1)
                              (for [i (range (count vector-data))
                                    j (range (inc i) (count vector-data))
                                    :when (< i j)]
                                (let [ch1 (nth vector-data i)
                                      ch2 (nth vector-data j)
                                      corr (calculate-correlation ch1 ch2)]
                                  {:channels [i j]
                                   :correlation corr})))
          _ (println "channel-coherence" channel-coherence)
          ; Combine all features
          features {:band-powers band-powers
                    :temporal-stats amplitude-stats
                    :channel-coherence channel-coherence
                    :timestamp (System/currentTimeMillis)
                    :channels (count vector-data)
                    :samples (if (> (count vector-data) 0)
                               (count (first vector-data))
                               0)
                    :duration (if (and (> (count vector-data) 0) (> sampling-rate 0))
                                (/ (count (first vector-data)) sampling-rate)
                                0)}]
_ (println "features" features)
      features)
    (catch Exception e
      (println "Error extracting signature features:" (.getMessage e))
      (.printStackTrace e)
      ;; Return basic fallback features
      {:band-powers {:delta 0.2 :theta 0.15 :alpha 0.25 :beta 0.3 :gamma 0.1}
       :timestamp (System/currentTimeMillis)})))

(defn create-signature-template
  "Create a template from signature features"
  [features]
  {:band-powers (:band-powers features)
   :temporal-patterns (mapv #(select-keys % [:mean :std-dev]) (:temporal-stats features))
   :coherence-patterns (mapv #(select-keys % [:channels :correlation]) (:channel-coherence features))
   :created-at (java.util.Date.)})

(defn process-wave-signature!
  "Process recorded EEG data into a wave signature"
  [eeg-data recording-dir metadata]
  (try
    (let [features (extract-signature-features eeg-data api/CURRENT_SRATE)
          feature-path (str recording-dir "/signature_features.edn")]
      (spit feature-path (pr-str features))
      (create-signature-template features))
    (catch Exception e
      (println "Error processing wave signature:" (.getMessage e))
      (.printStackTrace e))))

(defn stop-wave-signature-recording!
  "Stop recording a wave signature and process it"
  [category signature]
  (try
    (let [context @state/recording-context
          _ (println "CTXT STOPPING : " context)]

          ; Add end tag
      (let [timestamp (System/currentTimeMillis)
            _ (println "Category " category)]
        (swap! state/tags conj {:timestamp timestamp
                                :label (str "WAVE_SIGNATURE_END:"
                                            category "/" signature)}))

      (println "before context ends " context)
          ; Stop recording
      (record/stop-recording!)

          ; Process the recording into a wave signature
      (let [recording-dir (:lorfile-dir context)
            metadata (get-in context [:metadata])
            eeg-data @state/eeg-data
            profile-name (:name ((:get-active-profile @state/state)))
            include-in-aggregation (get metadata :include-in-aggregation true)

                ; Additional signature-specific metadata
            signature-metadata (merge metadata
                                      {:sample-count (count eeg-data)
                                       :tags @state/tags
                                       :processed-at (java.util.Date.)})]

        (spit (str recording-dir "/recording_metadata.edn")
              (pr-str signature-metadata))

              ; Save tags separately for easy access
        (spit (str recording-dir "/tags.edn")
              (pr-str @state/tags))

        (println "Count of samples " (count eeg-data))
              ; Process the signature features if we have enough data
        (if (>= (count eeg-data) MIN_SAMPLES_FOR_SIGNATURE)
          (do
            (process-wave-signature! eeg-data recording-dir signature-metadata)
            (println "Successfully processed wave signature for "
                     category "/" signature)

              ;; Auto-aggregate the category data
            (if include-in-aggregation
              (do
                (println "Attempting to aggregate category data for" category " for " (:name ((:get-active-profile @state/state))))
                (try
                    ;; Make sure we're using the correct namespace and function
                  (category/aggregate-after-recording! profile-name category signature)
                  (println "Category aggregation completed successfully")
                  (catch Exception agg-e
                    (println "Error during category aggregation:" (.getMessage agg-e))
                    (.printStackTrace agg-e))))
              (println "Skipping aggregation (include-in-aggregation is false)"))

            recording-dir)
          (do
            (println "Not enough data to process wave signature. Minimum required:"
                     MIN_SAMPLES_FOR_SIGNATURE)
            nil))))

      ;; Ensure recording is stopped in all cases
      (when @state/recording?
        (record/stop-recording!))
    (catch Exception e
      (println "Error stopping wave signature recording:" (.getMessage e))
      (.printStackTrace e)
      ;; Ensure recording is stopped even on error
      (when @state/recording?
        (record/stop-recording!))))
  nil)


(defn stop-category-recording!
  "Stop recording a wave signature category and process it"
  [category]
  (try
    (let [context @state/recording-context]

          ; Add end tag
      (let [timestamp (System/currentTimeMillis)
            _ (println "Category " category)]
        (swap! state/tags conj {:timestamp timestamp
                                :label (str "WAVE_SIGNATURE_END:"
                                            category)}))

      (println "before context ends " context)
          ; Stop recording
      (record/stop-recording!)

          ; Process the recording into a wave signature
      (let [category-recording-dir (:lorfile-dir context)
            metadata (get-in context [:metadata])
            eeg-data @state/eeg-data
            profile-name (:name ((:get-active-profile @state/state)))

                ; Additional signature-specific metadata
            signature-metadata (merge metadata
                                      {:sample-count (count eeg-data)
                                       :tags @state/tags
                                       :processed-at (java.util.Date.)})]

        (spit (str category-recording-dir "/recording_metadata.edn")
              (pr-str signature-metadata))

              ; Save tags separately for easy access
        (spit (str category-recording-dir "/tags.edn")
              (pr-str @state/tags))

        (println "Count of samples " (count eeg-data))
              ; Process the signature features if we have enough data
        (if (>= (count eeg-data) MIN_SAMPLES_FOR_SIGNATURE)
          (do
            (process-wave-signature! eeg-data category-recording-dir signature-metadata)
            (println "Successfully processed category recording for category: " category)

            category-recording-dir)
          (do
            (println "Not enough data to process category recording. Minimum required:"
                     MIN_SAMPLES_FOR_SIGNATURE)
            nil))))

    ;; Ensure recording is stopped in all cases
    (when @state/recording?
      (record/stop-recording!))
    (catch Exception e
      (println "Error stopping category recording:" (.getMessage e))
      (.printStackTrace e)
      ;; Ensure recording is stopped even on error
      (when @state/recording?
        (record/stop-recording!))))
  nil)

(defn add-wave-category
   "Interactive command to add a new wave category"
   []
   (try
     (print "Enter new category name: ")
     (flush)
     (let [category-name (read-line)]
       (when-not (str/blank? category-name)
         (let [config-path (str (fio/config-base-dir) "/config.edn")
               config (if (.exists (io/file config-path))
                        (edn/read-string (slurp config-path))
                        {})
               updated-config (update config :wave-lexicon-categories
                                      #(if (seq %)
                                         (conj % (keyword category-name))
                                         [(keyword category-name)]))]
           (spit config-path (pr-str updated-config)))

         (println "Created new category:" category-name)
         category-name))
     (catch Exception e
       (println "Error adding wave category:" (.getMessage e))
       nil)))

 (defn add-wave-signature
   "Interactive command to add a new wave signature"
   [board-shim]
   (if-not board-shim
     (println "No board connected. Please connect to a board first.")
     (try
      ; First, select a category
       (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
             categories (list-wave-signature-categories)]

         (println "\nAvailable categories:")
         (if (seq categories)
           (do
             (println (str "Found " (count categories) " lexicon categories:"))
             (doseq [cat categories]
               (println "  -" cat))
             (doall (map-indexed #(println (str (inc %1) ". " %2)) categories)))
           (println "No categories found."))

         (println (str (inc (count categories)) ". Create new category"))

         (print "\nSelect category number or enter new category name: ")
         (flush)
         (let [input (read-line)
               category (try
                          (let [idx (Integer/parseInt input)]
                            (if (<= idx (count categories))
                              (nth categories (dec idx))
                              nil))
                          (catch Exception _ nil))

              ; If input is a number and greater than categories count, create new category
               category-name (cond
                            ; Selected existing category
                               category
                               category

                            ; Create new category
                               (and (re-matches #"\d+" input)
                                    (= (Integer/parseInt input) (inc (count categories))))
                               (add-wave-category)

                            ; Input is a new category name
                               :else
                               input)]

           (when-not (str/blank? category-name)
            ; Now select or create a signature within that category
             (let [signatures (list-only-signatures profile-name category-name)]

               (println "\nAvailable signatures in category" category-name ":")
               (if (seq signatures)
                 (do
                   (println (str "Found " (count signatures) " signatures:"))
                   (doseq [sig signatures]
                     (println "  -" sig))
                   (doall (map-indexed #(println (str (inc %1) ". " %2)) signatures)))
                 (println "No signatures found."))

               (println (str (inc (count signatures)) ". Create new signature"))

               (print "\nSelect signature number or enter new signature name: ")
               (flush)
               (let [sig-input (read-line)
                     signature (try
                                 (let [idx (Integer/parseInt sig-input)]
                                   (if (<= idx (count signatures))
                                     (nth signatures (dec idx))
                                     nil))
                                 (catch Exception _ nil))

                      ; Determine signature name
                     signature-name (cond
                                   ; Selected existing signature
                                      signature
                                      signature

                                   ; Create new signature name
                                      (and (re-matches #"\d+" sig-input)
                                           (= (Integer/parseInt sig-input) (inc (count signatures))))
                                      (do
                                        (print "Enter new signature name: ")
                                        (flush)
                                        (read-line))

                                   ; Input is a new signature name
                                      :else
                                      sig-input)]

                 (when-not (str/blank? signature-name)
                      ; Ask if this is practice or real
                   (println "\nIs this a practice recording? (y/n)")
                   (println "Practice recordings will be saved but won't affect your aggregated signature data.")
                   (print "Enter choice (default: n): ")
                   (flush)
                   (let [practice? (= (str/lower-case (or (read-line) "n")) "y")
                         include-in-aggregation (not practice?)]

              ;; Ready to record
                     (println "\nStarting wave signature recording for:" category-name "/" signature-name)
                     (if practice?
                       (println "PRACTICE MODE: This recording will not be included in aggregation.")
                       (println "RECORD MODE: This recording will be included in aggregation."))
                     (println "Focus on the mental state you want to capture.")
                     (println "Press any key when ready to start recording...")
                     (read-line)

                     (let [context (start-wave-signature-recording!
                                    category-name signature-name
                                    :include-in-aggregation include-in-aggregation)]
                       (if context
                         (do
                           (println "Recording started. Please focus...")
                           (println "Recording will automatically stop after 5 seconds, or press any key to stop sooner...")

                        ; Timed stop or manual stop, whichever comes first
                           (let [start-time (System/currentTimeMillis)
                                 future-stop (future
                                               (Thread/sleep DEFAULT_RECORDING_DURATION)
                                               (println "Automatic recording stop after 5 seconds")
                                               (stop-wave-signature-recording! category-name signature-name))]

                          ; Wait for user input
                             (read-line)

                          ; Cancel the automatic stop if user pressed key
                             (future-cancel future-stop)

                          ; Stop recording if not already stopped
                             (when @state/recording?
                               (println "Manual recording stop")
                               (stop-wave-signature-recording! category-name signature-name))

                             (println "Wave signature recorded and saved successfully.")))
                         (println "Failed to start recording. Check your device connection."))))))))))
   (catch Exception e
     (println "Error in add-wave-signature:" (.getMessage e))
     (.printStackTrace e)))))

(defn load-category-template
  "Load a category template for matching"
  [profile-name category]
  (try
    (let [category-dir (fio/get-wave-lexicon-dir profile-name category)
          template-file (str category-dir "/" category "/category_template.edn")]

      (when (.exists (io/file template-file))
        (edn/read-string (slurp template-file))))
    (catch Exception e
      (println "Error loading category template:" (.getMessage e))
      nil)))

(defn calculate-temporal-similarity
  "Calculate similarity between temporal patterns"
  [current-stats template-patterns]
  (try
    (let [channel-similarities (for [i (range api/CURRENT_CHANNEL_COUNT)
                                     :let [current-chan (nth current-stats i)
                                           template-chan (nth template-patterns i)]]
                                 (let [mean-diff (Math/abs (- (:mean current-chan)
                                                              (:mean template-chan)))
                                       std-diff (Math/abs (- (:std-dev current-chan)
                                                             (:std-dev template-chan)))
                                      ; Normalize differences
                                       mean-sim (/ 1.0 (+ 1.0 mean-diff))
                                       std-sim (/ 1.0 (+ 1.0 std-diff))]
                                  ; Average for this channel
                                   (/ (+ mean-sim std-sim) 2.0)))
          ;; Average across channels
          overall-similarity (if (pos? api/CURRENT_CHANNEL_COUNT)
                               (/ (reduce + channel-similarities) api/CURRENT_CHANNEL_COUNT)
                               0.5)]
      overall-similarity)
    (catch Exception e
      (println "Error calculating temporal similarity:" (.getMessage e))
      0.5)))

(defn calculate-coherence-similarity
  "Calculate similarity between coherence patterns"
  [current-coherence template-coherence]
  (try
    (if (or (empty? current-coherence) (empty? template-coherence))
      0.5 ; Default value if no coherence data

      (let [; Create maps for easier lookup
            current-map (reduce (fn [acc item]
                                  (assoc acc (:channels item) (:correlation item)))
                                {}
                                current-coherence)

            template-map (reduce (fn [acc item]
                                   (assoc acc (:channels item) (:correlation item)))
                                 {}
                                 template-coherence)

            ; Find common channel pairs
            all-pairs (into #{} (concat (keys current-map) (keys template-map)))

            ; Calculate differences
            differences (for [pair all-pairs
                              :let [current-val (get current-map pair 0.0)
                                    template-val (get template-map pair 0.0)
                                    diff (Math/abs (- current-val template-val))]]
                          diff)

            ; Average difference
            avg-diff (if (pos? (count differences))
                       (/ (reduce + differences) (count differences))
                       1.0)

            ; Convert to similarity
            similarity (/ 1.0 (+ 1.0 avg-diff))]

        similarity))
    (catch Exception e
      (println "Error calculating coherence similarity:" (.getMessage e))
      0.5)))

(defn calculate-band-power-similarity
  "Calculate similarity between band power distributions"
  [current-bands template-bands]
  (try
    (let [bands [:delta :theta :alpha :beta :gamma]

          ; Calculate Euclidean distance
          squared-diffs (for [band bands
                              :let [current-val (get current-bands band 0.0)
                                    template-val (get template-bands band 0.0)
                                    diff (- current-val template-val)]]
                          (* diff diff))

          distance (Math/sqrt (reduce + squared-diffs))

          ; Convert to similarity (0-1 range)
          similarity (/ 1.0 (+ 1.0 distance))]

      similarity)
    (catch Exception e
      (println "Error calculating band power similarity:" (.getMessage e))
      0.5)))

(defn calculate-similarity
  "Calculate similarity between two feature sets"
  [current-features template-features]
  (try
    (let [; Extract band powers for comparison
          current-powers (get-in current-features [:band-powers])
          template-powers (get-in template-features [:band-powers])

          ; Calculate Euclidean distance between power spectra
          band-keys (into #{} (concat (keys current-powers) (keys template-powers)))
          power-distance-sq (reduce +
                                    (for [k band-keys]
                                      (let [curr-val (get current-powers k 0.0)
                                            template-val (get template-powers k 0.0)
                                            diff (- curr-val template-val)]
                                        (* diff diff))))

          ; Compare temporal patterns (if available)
          current-temporal (get-in current-features [:temporal-stats])
          template-temporal (get-in template-features [:temporal-patterns])

          temporal-distance (when (and current-temporal template-temporal)
                              (reduce +
                                      (for [i (range (min (count current-temporal)
                                                          (count template-temporal)))]
                                        (let [curr-stats (get current-temporal i)
                                              temp-stats (get template-temporal i)
                                              mean-diff (- (get curr-stats :mean 0)
                                                           (get temp-stats :mean 0))
                                              std-diff (- (get curr-stats :std-dev 0)
                                                          (get temp-stats :std-dev 0))]
                                          (+ (* mean-diff mean-diff) (* std-diff std-diff))))))

          ; Combined distance
          total-distance (+ power-distance-sq (or temporal-distance 0))

          ; Convert to similarity score (0-1)
          similarity (/ 1.0 (+ 1.0 (Math/sqrt total-distance)))]

      similarity)
    (catch Exception e
      (println "Error calculating similarity:" (.getMessage e))
      0.0)))

(defn calculate-signature-similarity
  "Calculate similarity between current features and a template"
  [current-features template]
  (try
    (let [; Calculate band power similarity (most important)
          band-similarity (calculate-band-power-similarity
                           (:band-powers current-features)
                           (:band-powers template))

          ; Calculate temporal pattern similarity
          temporal-similarity (if (and (:temporal-stats current-features)
                                       (:temporal-patterns template))
                                (calculate-temporal-similarity
                                 (:temporal-stats current-features)
                                 (:temporal-patterns template))
                                0.5) ; Default if not available

          ; Calculate coherence pattern similarity
          coherence-similarity (if (and (:channel-coherence current-features)
                                        (:coherence-patterns template))
                                 (calculate-coherence-similarity
                                  (:channel-coherence current-features)
                                  (:coherence-patterns template))
                                 0.5) ; Default if not available

          ; Weighted combination (band powers most important)
          combined-similarity (+ (* 0.6 band-similarity)
                                 (* 0.2 temporal-similarity)
                                 (* 0.2 coherence-similarity))]

      combined-similarity)
    (catch Exception e
      (println "Error calculating signature similarity:" (.getMessage e))
      0.0)))

(defn similarity-score
  "Calculate similarity score between current features and a template"
  [features template]
  (try
    (let [; Compare band powers (main feature)
          bp-current (:band-powers features)
          bp-template (:band-powers template)

          ; Calculate similarity for each band
          band-scores (for [band [:delta :theta :alpha :beta :gamma]]
                        (let [c (get bp-current band 0.0)
                              t (get bp-template band 0.0)
                              diff (Math/abs (- c t))
                              max-val (max 0.0001 (max c t))  ; Avoid division by zero
                              similarity (- 1.0 (min 1.0 (/ diff max-val)))]
                          similarity))

          ; Average band power similarity
          avg-band-score (/ (apply + band-scores) (count band-scores))

          ; Could add more feature comparisons here

          ; Overall score (can adjust weights)
          overall-score avg-band-score]

      overall-score)
    (catch Exception e
      (println "Error calculating similarity:" (.getMessage e))
      0.0)))

(defn calculate-feature-distance
  "Calculate distance between two feature sets"
  [features1 features2]
  (try
    (let [; Calculate band power distance
          band-dist (let [bp1 (:band-powers features1)
                          bp2 (:band-powers features2)
                          keys (keys bp1)]
                      (/ (reduce + (for [k keys
                                         :let [v1 (get bp1 k 0)
                                               v2 (get bp2 k 0)]]
                                     (Math/pow (- v1 v2) 2)))
                         (count keys)))

          ; Calculate temporal pattern distance
          temp-dist (let [tp1 (:temporal-patterns features1)
                          tp2 (:temporal-patterns features2)]
                      (if (and tp1 tp2 (= (count tp1) (count tp2)))
                        (/ (reduce + (for [i (range (count tp1))
                                           :let [t1 (nth tp1 i)
                                                 t2 (nth tp2 i)]]
                                       (+ (Math/pow (- (:mean t1) (:mean t2)) 2)
                                          (Math/pow (- (:std-dev t1) (:std-dev t2)) 2))))
                           (count tp1))
                        1.0))

          ; Calculate coherence pattern distance (simplified)
          coh-dist (let [cp1 (:coherence-patterns features1)
                         cp2 (:coherence-patterns features2)]
                     (if (and cp1 cp2 (= (count cp1) (count cp2)))
                       (/ (reduce + (for [i (range (count cp1))
                                          :let [c1 (nth cp1 i)
                                                c2 (nth cp2 i)]]
                                      (Math/pow (- (:correlation c1) (:correlation c2)) 2)))
                          (count cp1))
                       1.0))

          ; Weighted total distance
          total-dist (+ (* 0.5 band-dist)
                        (* 0.3 temp-dist)
                        (* 0.2 coh-dist))

          ; Convert to similarity score (0-1)
          similarity (max 0.0 (min 1.0 (- 1.0 (/ total-dist 3.0))))]
      similarity)
    (catch Exception e
      (println "Error calculating feature distance:" (.getMessage e))
      0.0)))

(defn match-against-category
  "Match current EEG data against a category template"
  [eeg-data profile-name category]
  (try
    (let [category-dir (fio/get-wave-lexicon-dir profile-name category)
          template-file (str category-dir "/category_template.edn")]

      (if (.exists (io/file template-file))
        (let [template (edn/read-string (slurp template-file))
              sampling-rate (brainflow/get-sampling-rate (brainflow/get-board-id @state/shim))
              features (extract-signature-features eeg-data sampling-rate)
              feature-template (create-signature-template features)
              similarity (calculate-feature-distance feature-template template)]

          {:category category
           :confidence similarity
           :timestamp (System/currentTimeMillis)})

        {:category category
         :confidence 0.0
         :timestamp (System/currentTimeMillis)}))
    (catch Exception e
      (println "Error matching against category:" (.getMessage e))
      {:category category
       :confidence 0.0
       :timestamp (System/currentTimeMillis)
       :error (.getMessage e)})))

(defn match-live-activity
  "Match current brain activity against wave signatures"
  [current-data]
  (try
    (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
          normalized-data (stream/normalize-data-format current-data)
          current-features (extract-signature-features normalized-data api/CURRENT_SRATE)
          categories (list-wave-signature-categories)]

      (if (empty? categories)
        (do
          (println "No wave signature categories available")
          nil)

        ; Calculate similarity with each category template
        (let [similarities (for [category categories
                                 :let [template (load-category-template profile-name category)]
                                 :when template]
                             {:category category
                              :similarity (calculate-signature-similarity
                                           current-features
                                           template)})

              ; Sort by similarity (highest first)
              sorted-similarities (sort-by :similarity > similarities)

              ; Get best match above threshold
              best-match (first (filter #(>= (:similarity %) SIMILARITY_THRESHOLD)
                                        sorted-similarities))]
          (if best-match
            (do
              (println "Detected activity:" (:category best-match)
                       "with confidence" (:similarity best-match))
              best-match)
            (do
              (println "No matching activity detected")
              nil)))))
    (catch Exception e
      (println "Error matching live activity:" (.getMessage e))
      (.printStackTrace e)
      nil)))


(defn train-model
  "Train a model using collected wave signatures"
  []
  (try
    (let [profile-name (or (:name ((:get-active-profile @state/state))) "default")
          categories (list-wave-signature-categories)]

      (if (empty? categories)
        (println "No wave signature categories found. Please add signatures first.")

        (do
          (println "Training model using the following categories:")
          (doseq [category categories]
            (let [stats (category/get-category-stats profile-name category)]
              (println (format "  - %s (%d signatures%s)"
                               category
                               (:signature-count stats)
                               (if (:has-template stats) ", has template" "")))))

          ; Here we would implement actual model training
          ; For now, we'll just update all the category templates
          (doseq [category categories]
            (println "\nUpdating template for category:" category)

            (let [dir (fio/get-wave-lexicon-dir profile-name category)
                  signatures (when (.exists (io/file dir))
                               (->> (.listFiles (io/file dir))
                                    (filter #(.isDirectory %))
                                    (filter #(not= (.getName %) "category_template.edn"))))

                  ; Collect and combine all signatures
                  combined-features (atom nil)]

              ; Process each signature
              (doseq [sig-dir signatures]
                (let [feature-file (io/file (str (.getPath sig-dir) "/signature_features.edn"))]
                  (when (.exists feature-file)
                    (let [features (edn/read-string (slurp feature-file))
                          template (create-signature-template features)]

                      ; Combine with existing data
                      (if @combined-features
                        ; Update combined features with this signature
                        (reset! combined-features
                                (merge-with (fn [a b]
                                              (cond
                                                (and (map? a) (map? b))
                                                (merge-with + a b)

                                                (and (vector? a) (vector? b))
                                                (mapv (fn [x y]
                                                        (if (map? x)
                                                          (merge-with + x y)
                                                          (+ x y)))
                                                      a b)

                                                :else (+ a b)))
                                            @combined-features
                                            template))
                        ; First signature becomes the base
                        (reset! combined-features template))))))

              ; If we have combined data, create a normalized template
              (when @combined-features
                (let [sig-count (count signatures)]
                  (when (pos? sig-count)
                    ; Normalize values
                    (let [normalized (clojure.walk/postwalk
                                      (fn [x]
                                        (if (number? x)
                                          (/ x sig-count)
                                          x))
                                      @combined-features)

                          ;; Add metadata
                          final-template (assoc normalized
                                                :instance-count sig-count
                                                :last-updated (java.util.Date.)
                                                :category category)]

                      ; Save the updated template
                      (spit (str dir "/category_template.edn") (pr-str final-template))
                      (println "  Updated template with" sig-count "signatures")))))))
          (println "\nModel training complete!"))))
    (catch Exception e
      (println "Error training model:" (.getMessage e))
      (.printStackTrace e))))

(defn initialize-lexicon! []
  (state/register-fn! :start-wave-signature-recording!  start-wave-signature-recording!)
  (state/register-fn! :stop-wave-signature-recording!   stop-wave-signature-recording!)
  (state/register-fn! :list-all-wave-signatures         list-all-wave-signatures)
  (state/register-fn! :list-wave-signature-categories   list-wave-signature-categories)
  (state/register-fn! :list-wave-signatures-by-category list-wave-signatures-by-category)
  (state/register-fn! :save-wave-lexicon-entry!         save-wave-lexicon-entry!)
  (state/register-fn! :match-live-activity              match-live-activity)
  (state/register-fn! :get-category-stats               category/get-category-stats)
  (state/register-fn! :add-wave-signature               add-wave-signature)
  (state/register-fn! :train-model                      train-model))