(ns reveal.core
  (:require [clojure.core.server :as server]
            [clojure.core.async :as a :refer [<!]]
            [cljfx.api :as fx]
            [reveal.font :as font]
            [clojure.main :as m]
            [reveal.stream :as stream]
            [reveal.canvas :as canvas]
            [reveal.event :as event]
            [reveal.parse :as parse])
  (:import [javafx.event Event]))

(defn- root-view [{:keys [showing canvas]}]
  {:fx/type :stage
   :on-close-request #(.consume ^Event %)
   :showing showing
   :width 400
   :height 500
   :on-focused-changed {::event/type ::canvas/on-window-focus-changed}
   :scene {:fx/type :scene
           :root {:fx/type :v-box
                  :style {:-fx-background-color "#4a4a4a"}
                  :children [{:fx/type canvas/view
                              :v-box/vgrow :always
                              :layout canvas}]}}})

(defn oneduce
  ([xf x]
   (oneduce xf identity x))
  ([xf f x]
   (let [rf (xf (completing #(f %2)))]
     (rf (rf nil x)))))

(defn- execute-effect [action ch]
  (future
    (try
      (let [ret ((:invoke action))]
        (a/put! ch (parse/action-output action ret)))
      (catch Exception e
        (a/put! ch (parse/action-error-output action e))))))

(defn start [& {:keys [prepl]
                :or {prepl server/prepl}}]
  (let [out *out*
        err *err*
        ch (a/chan 1000)
        font (font/make {:family "Fantasque Sans Mono" :size 15})
        stream-xf (comp (stream/stream-xf font)
                        (partition-all 128))
        *state (atom {:canvas (canvas/layout {:font font
                                              :lines []
                                              :canvas-width 0.0
                                              :canvas-height 0.0
                                              :scroll-x 0.0
                                              :scroll-y 0.0})
                      :showing true})
        event-handler (-> event/handle
                          (fx/wrap-co-effects
                            {:state (fx/make-deref-co-effect *state)})
                          (fx/wrap-effects
                            {:dispatch fx/dispatch-effect
                             :state (fx/make-reset-effect *state)
                             :execute (fn [action _]
                                        (execute-effect action ch))})
                          (fx/wrap-async :error-handler #(binding [*out* err]
                                                           (prn %2))))
        renderer (fx/create-renderer
                   :opts {:fx.opt/map-event-handler event-handler}
                   :middleware (fx/wrap-map-desc #(root-view %)))]
    (a/go-loop [x (<! ch)]
      (when (some? x)
        (oneduce stream-xf #(event-handler {::event/type ::canvas/add-lines :fx/event %}) x)
        (recur (<! ch))))
    (fx/mount-renderer *state renderer)
    (prepl *in* (fn [x]
                  (a/put! ch (parse/prepl-output x))
                  (binding [*out* out
                            *print-length* 5]
                    (if (:exception x)
                      (binding [*out* err]
                        (println (m/ex-str (m/ex-triage (:val x))))
                        (flush))
                      (do
                        (case (:tag x)
                          :out (println (:val x))
                          :err (binding [*out* err]
                                 (println (:val x)))
                          (:tap :ret) (prn (:val x))
                          (prn x))
                        (flush)))
                    (print (str (:ns x) "=> "))
                    (flush))))
    (fx/unmount-renderer *state renderer)
    nil))

#_(start)

#_(/ 1 0)

#_(range 10)

#_(do :repl/quit)

(defn -main [] (start))
