(ns reveal.canvas
  (:require [cljfx.lifecycle :as fx.lifecycle]
            [cljfx.mutator :as fx.mutator]
            [cljfx.fx.canvas :as fx.canvas]
            [cljfx.api :as fx]
            [reveal.actions :as actions]
            [cljfx.composite :as fx.composite]
            [reveal.font :as font]
            [reveal.cursor :as cursor]
            [reveal.lines :as lines]
            [reveal.event :as event]
            [clojure.string :as str])
  (:import [javafx.scene.canvas Canvas GraphicsContext]
           [javafx.scene.paint Color]
           [javafx.scene.input ScrollEvent MouseEvent MouseButton KeyEvent KeyCode Clipboard ClipboardContent]
           [javafx.stage Popup Screen]
           [javafx.geometry BoundingBox Rectangle2D]
           [javafx.event EventDispatcher Event]
           [com.sun.javafx.event RedirectedEvent]))

(set! *warn-on-reflection* true)

(defn- make-resizable-canvas []
  (proxy [Canvas] []
    (isResizable [] true)
    (minWidth [_] 0)
    (minHeight [_] 0)
    (maxWidth [_] Double/MAX_VALUE)
    (maxHeight [_] Double/MAX_VALUE)
    (prefWidth [_]
      (let [^Canvas this this]
        (.getWidth this)))
    (prefHeight [_]
      (let [^Canvas this this]
        (.getHeight this)))
    (resize [w h]
      (let [^Canvas this this]
        (proxy-super setWidth w)
        (proxy-super setHeight h)))))

(def ^:private canvas-lifecycle
  (fx.composite/lifecycle
    {:ctor make-resizable-canvas
     :args []
     :props (merge fx.canvas/props
                   (fx.composite/props Canvas
                     :draw [(fx.mutator/setter
                              (fn [^Canvas canvas [f & args]]
                                (apply f (.getGraphicsContext2D canvas) args)))
                            fx.lifecycle/scalar]
                     :on-width-changed [:property-change-listener fx.lifecycle/change-listener]
                     :on-height-changed [:property-change-listener fx.lifecycle/change-listener]
                     :popup [(fx.mutator/adder-remover
                               (fn [^Canvas canvas ^Popup popup]
                                 (.show popup (.getWindow (.getScene canvas))))
                               (fn [_ ^Popup popup]
                                 (.hide popup)))
                             fx.lifecycle/dynamic]))}))

(defn- clamp [n min-n max-n]
  (-> n
      (max min-n)
      (min max-n)))

(def ^:private ^:const scroll-bar-breadth 10.0)

(def ^:private ^:const min-scroll-tab-size 30.0)

(defn- scroll-tab-size [document-size canvas-size]
  (if (< document-size min-scroll-tab-size)
    min-scroll-tab-size
    (let [visible-ratio (min 1.0 (/ canvas-size document-size))]
      (max min-scroll-tab-size (* canvas-size visible-ratio)))))

(defn- scrolled-to-bottom? [layout]
  (let [{:keys [scroll-y
                canvas-height
                document-height]} layout]
    (or (< document-height canvas-height)
        (= scroll-y (- canvas-height document-height)))))

(defn- scroll-per-pixel [document-size canvas-size scroll-tab-size]
  (let [document-range (- document-size canvas-size)
        scroll-range (- canvas-size scroll-tab-size)]
    (if (zero? scroll-range)
      ##Inf
      (/ document-range scroll-range))))

(defn layout [{:keys [font
                      canvas-width
                      canvas-height
                      lines
                      scroll-x
                      scroll-y] :as layout}]
  (let [line-height (font/line-height font)
        line-count (count lines)
        document-height (+ (* line-height line-count) scroll-bar-breadth)
        scroll-y (clamp scroll-y (- canvas-height document-height) 0.0)
        scroll-y-remainder (rem (- scroll-y) line-height)
        dropped-line-count (- (long (/ scroll-y line-height)))
        drawn-line-count (long (min (- line-count dropped-line-count)
                                    (Math/ceil (/ (+ canvas-height scroll-y-remainder) line-height))))
        document-width (transduce
                         (map #(transduce (comp (mapcat :segments) (map :width)) + 0 (lines %)))
                         max
                         0
                         (range dropped-line-count (+ dropped-line-count drawn-line-count)))
        scroll-x (clamp scroll-x (- canvas-width document-width) 0.0)]
    (-> layout
        (assoc :scroll-x scroll-x
               :scroll-y scroll-y
               :document-height document-height
               :drawn-line-count drawn-line-count
               :dropped-line-count dropped-line-count
               :scroll-y-remainder scroll-y-remainder)
        (as-> $ (if (>= canvas-width document-width)
                  (dissoc $ :scroll-tab-x)
                  (assoc $ :scroll-tab-x (let [visible-left (- scroll-x)
                                               scroll-tab-top (- canvas-height scroll-bar-breadth)
                                               scroll-tab-width (scroll-tab-size document-width canvas-width)
                                               scroll-per-pixel (scroll-per-pixel document-width canvas-width scroll-tab-width)
                                               scroll-tab-left (/ visible-left scroll-per-pixel)]
                                           {:x scroll-tab-left
                                            :y scroll-tab-top
                                            :width scroll-tab-width
                                            :height scroll-bar-breadth
                                            :scroll-per-pixel scroll-per-pixel}))))
        (as-> $ (if (>= canvas-height document-height)
                  (dissoc $ :scroll-tab-y)
                  (assoc $ :scroll-tab-y (let [visible-top (- scroll-y)
                                               scroll-tab-left (- canvas-width scroll-bar-breadth)
                                               scroll-tab-height (scroll-tab-size document-height canvas-height)
                                               scroll-per-pixel (scroll-per-pixel document-height canvas-height scroll-tab-height)
                                               scroll-tab-top (/ visible-top scroll-per-pixel)]
                                           {:x scroll-tab-left
                                            :y scroll-tab-top
                                            :width scroll-bar-breadth
                                            :height scroll-tab-height
                                            :scroll-per-pixel scroll-per-pixel})))))))

(defn- draw-scroll-bar [^GraphicsContext ctx active {:keys [x y width height]}]
  (doto ctx
    (.setFill (Color/valueOf (if active "#fff6" "#eee3")))
    (.fillRoundRect x y width height scroll-bar-breadth scroll-bar-breadth)))

(defn- draw [^GraphicsContext ctx layout]
  (let [{:keys [canvas-height
                canvas-width
                scroll-y-remainder
                ^long drawn-line-count
                ^long dropped-line-count
                scroll-x
                font
                lines
                scroll-tab-x
                scroll-tab-y
                cursor
                anchor
                gesture]} layout
        line-height (font/line-height font)]
    (.clearRect ctx 0 0 canvas-width canvas-height)
    (when (and cursor anchor)
      (let [from (-> anchor
                     (cursor/min cursor)
                     (cursor/max [dropped-line-count 0]))
            last-visible-line-index (dec (+ dropped-line-count drawn-line-count))
            to (-> anchor
                   (cursor/max cursor)
                   (cursor/min [last-visible-line-index (dec (count (lines last-visible-line-index)))]))]
        (when-not (cursor/before? to from)
          (.setFill ctx (Color/valueOf "#666"))
          (doseq [i (range (cursor/row from) (inc (cursor/row to)))]
            (let [line (lines i)
                  start-col (if (= i (cursor/row from))
                              (cursor/col from)
                              0)
                  end-col (if (= i (cursor/row to))
                            (cursor/col to)
                            (dec (count line)))
                  x (transduce
                      (comp
                        (take start-col)
                        (mapcat :segments)
                        (map :width))
                      +
                      scroll-x
                      line)
                  width (transduce
                          (comp
                            (drop start-col)
                            (take (inc (- end-col start-col)))
                            (mapcat :segments)
                            (map :width))
                          +
                          0
                          line)
                  y (- (* line-height (- i dropped-line-count))
                       scroll-y-remainder)]
              (.fillRect ctx x y width line-height))))))
    (dotimes [i drawn-line-count]
      (transduce (mapcat :segments)
                 (completing
                   (fn [x {:keys [text width style]}]
                     (if (< x canvas-width)
                       (let [end (+ x width)]
                         (if (<= end 0)
                           end
                           (do
                             (.setFill ctx (Color/valueOf ^String (:fill style "#000")))
                             (.setFont ctx (font/jfx-font font (:font style)))
                             (.fillText ctx text x (-> (* i line-height)
                                                       (+ (font/ascent font (:font style)))
                                                       (- scroll-y-remainder)))
                             end)))
                       (reduced nil))))
                 scroll-x
                 (lines (+ i dropped-line-count))))
    (some->> scroll-tab-x (draw-scroll-bar ctx (= :scroll-x (:type gesture))))
    (some->> scroll-tab-y (draw-scroll-bar ctx (= :scroll-y (:type gesture))))))

(defn- set-cursor
  "Set cursor

  - `:anchor` - either true/false, or specific cursor value
  - `:align` - whether should update align char index used for vertical navigation"
  [layout cursor & {:keys [anchor align]
                    :or {anchor true
                         align true}}]
  (-> layout
      (assoc :cursor cursor)
      (cond-> (or align (nil? (:align-char-index layout)))
              (assoc :align-char-index (:index (get-in (:lines layout) cursor)))

              (or anchor (nil? (:anchor layout)))
              (assoc :anchor (if (cursor/cursor? anchor) anchor cursor)))))

(defn- remove-cursor [layout]
  (dissoc layout :cursor :anchor :align-char-index))

(defn- update-layout-fx [state path f & args]
  {:state (apply update-in state path (comp layout f) args)})

(defn- scroll-by [layout dx dy]
  (-> layout
      (update :scroll-x + dx)
      (update :scroll-y + dy)))

(defmethod event/handle ::on-scroll [{:keys [state path ^ScrollEvent fx/event]}]
  (update-layout-fx state path scroll-by (.getDeltaX event) (.getDeltaY event)))

(defmethod event/handle ::on-size-changed [{:keys [state path key fx/event]}]
  (update-layout-fx state path assoc key event))

(defn- add-lines [layout lines]
  (-> layout
      (update :lines into lines)
      (cond-> (scrolled-to-bottom? layout)
              (assoc :scroll-y ##-Inf))))

(defmethod event/handle ::add-lines [{:keys [state path fx/event]}]
  (update-layout-fx state path add-lines event))

(defmethod event/handle ::on-mouse-released [{:keys [state path]}]
  (update-layout-fx state path dissoc :gesture))

(defmethod event/handle ::on-window-focus-changed [{:keys [fx/event state path]}]
  (when-not event (update-layout-fx state path dissoc :gesture)))

(defn- perform-scroll [layout ^MouseEvent event]
  (if-let [gesture (:gesture layout)]
    (case (:type gesture)
      :scroll-x
      (assoc layout :scroll-x (- (* (- (.getX event) (:offset gesture))
                                    (-> layout :scroll-tab-x :scroll-per-pixel))))
      :scroll-y
      (assoc layout :scroll-y (- (* (- (.getY event) (:offset gesture))
                                    (-> layout :scroll-tab-y :scroll-per-pixel))))

      layout)
    layout))

(defmethod event/handle ::on-mouse-dragged [{:keys [fx/event state path]}]
  (update-layout-fx state path perform-scroll event))

(defn- empty-region? [region]
  (every? #(-> % :text str/blank?) (:segments region)))

(defn- region-width [region]
  (transduce (map :width) + (:segments region)))

(defn- canvas->cursor [layout x y]
  (let [{:keys [scroll-x scroll-y font lines]} layout
        doc-x (- x scroll-x)
        doc-y (- y scroll-y)
        line-height (font/line-height font)
        row (long (/ doc-y line-height))]
    (when (< row (count lines))
      (let [line (lines row)
            index (first (transduce
                           (map #(region-width (line %)))
                           (completing
                             (fn [[i x] width]
                               (let [x (+ x width)]
                                 (if (< doc-x x)
                                   (reduced [i])
                                   [(inc i) x]))))
                           [0 0]
                           (range (count line))))]
        (when (and (< index (count line)) (not (empty-region? (line index))))
          [row index])))))

(defn- cursor->canvas-bounds ^BoundingBox [layout]
  (let [{:keys [lines cursor scroll-x scroll-y font]} layout
        line-height (font/line-height font)
        [row col] cursor
        line (lines row)]
    (BoundingBox. (+ scroll-x (transduce (map #(-> % line region-width)) + (range col)))
                  (double (+ scroll-y (* line-height row)))
                  (region-width (line col))
                  line-height)))

(defn- adjust-scroll [scroll canvas-size region-start region-size]
  (let [canvas-start (- scroll)
        region-end (+ region-start region-size)
        canvas-end (+ canvas-start canvas-size)
        start (if (> region-end canvas-end)
                (- region-start (- region-end canvas-end))
                region-start)
        start (if (< start canvas-start)
                (+ start (- canvas-start start))
                start)]
    (- scroll (- region-start start))))

(defn- ensure-cursor-visible [layout]
  (let [{:keys [lines cursor canvas-width canvas-height font]} layout
        [row col] cursor
        line (lines row)
        line-height (font/line-height font)
        region-size (region-width (line col))
        region-start (transduce (map region-width) + (subvec line 0 col))]
    (-> layout
        (update :scroll-y adjust-scroll canvas-height (* line-height (cursor/row cursor)) line-height)
        (update :scroll-x adjust-scroll canvas-width region-start region-size))))

(defn- show-popup [layout ^KeyEvent event]
  (let [layout (ensure-cursor-visible layout)
        {:keys [lines cursor]} layout
        actions (actions/collect (:values (get-in lines cursor)))
        bounds (cursor->canvas-bounds layout)
        ^Canvas target (.getTarget event)
        screen-bounds (.localToScreen target bounds)]
    (cond-> layout
            (pos? (count actions))
            (assoc :popup {:actions actions
                           :bounds screen-bounds}))))

(defn- handle-mouse-pressed [layout ^MouseEvent event]
  (cond
    (= (.getButton event) MouseButton/SECONDARY)
    (if-let [cursor (canvas->cursor layout (.getX event) (.getY event))]
      (-> layout
          (set-cursor cursor)
          (show-popup event))
      layout)

    (not= (.getButton event) MouseButton/PRIMARY)
    layout

    (some-> (:scroll-tab-y layout) :x (<= (.getX event)))
    (-> layout
        (assoc :gesture {:type :scroll-y
                         :offset (let [event-y (.getY event)
                                       {:keys [y height]} (:scroll-tab-y layout)]
                                   (if (<= y event-y (+ y height))
                                     (- event-y y)
                                     (* height 0.5)))})
        (perform-scroll event))

    (some-> (:scroll-tab-x layout) :y (<= (.getY event)))
    (-> layout
        (assoc :gesture {:type :scroll-x
                         :offset (let [event-x (.getX event)
                                       {:keys [x width]} (:scroll-tab-x layout)]
                                   (if (<= x event-x (+ x width))
                                     (- event-x x)
                                     (* width 0.5)))})
        (perform-scroll event))

    :else
    (if-let [cursor (canvas->cursor layout (.getX event) (.getY event))]
      (set-cursor layout cursor)
      layout)))

(defmethod event/handle ::on-mouse-pressed [{:keys [fx/event state path]}]
  (update-layout-fx state path handle-mouse-pressed event))

(defn- arrow-scroll [layout size-key]
  (let [{:keys [font]} layout
        line-height (font/line-height font)
        size (get layout size-key)]
    (* line-height
       (-> 5
           (min (Math/ceil (* 0.1 (/ size line-height))))
           (max 1)))))

(defn- page-scroll [layout]
  (let [{:keys [font canvas-height]} layout
        line-height (font/line-height font)]
    (* line-height
       (max 1 (Math/ceil (* 0.5 (/ canvas-height line-height)))))))

(def non-empty-region? (complement empty-region?))

(defn- empty-line? [line]
  (every? empty-region? line))

(def non-empty-line? (complement empty-line?))

(defn- introduce-cursor-at-bottom-of-screen [layout]
  (let [{:keys [drawn-line-count dropped-line-count lines canvas-height scroll-y-remainder font]} layout
        start-row (cond-> (dec (+ dropped-line-count drawn-line-count))
                          (< canvas-height (- (* drawn-line-count (font/line-height font))
                                              scroll-y-remainder))
                          dec)]
    (if-let [cursor (lines/scan lines [start-row -1] dec inc non-empty-region?)]
      (-> layout
          (set-cursor cursor)
          ensure-cursor-visible)
      layout)))

(defn- introduce-cursor-at-top-of-screen [layout]
  (let [{:keys [dropped-line-count lines scroll-y-remainder]} layout
        start-row (cond-> dropped-line-count
                          (not (zero? scroll-y-remainder))
                          inc)]
    (if-let [cursor (lines/scan lines [start-row -1] inc inc non-empty-region?)]
      (-> layout
          (set-cursor cursor)
          ensure-cursor-visible)
      layout)))

(defn- move-cursor-horizontally [layout with-anchor direction]
  (let [{:keys [cursor lines]} layout]
    (if-let [cursor (lines/scan lines cursor direction direction non-empty-region?)]
      (-> layout
          (set-cursor cursor :anchor with-anchor)
          ensure-cursor-visible)
      layout)))

(defn- select-all [layout]
  (let [{:keys [lines]} layout
        from (lines/scan lines [##-Inf ##-Inf] inc inc non-empty-region?)
        to (lines/scan lines [##Inf ##Inf] dec dec non-empty-region?)]
    (cond-> layout
            (and from to)
            (set-cursor to :anchor from))))

(defn- binary-nearest-by [f xs x]
  (let [last-i (dec (count xs))]
    (loop [low 0
           high last-i]
      (when (<= low high)
        (let [i (quot (+ low high) 2)
              n (f (xs i))]
          (cond
            (and (<= n x)
                 (or (= i last-i)
                     (< x (f (xs (inc i))))))
            i

            (< x n)
            (recur low (dec i))

            :else
            (recur (inc i) high)))))))

(defn- move-cursor-vertically [layout with-anchor direction]
  (let [{:keys [cursor lines align-char-index]} layout
        row (cursor/row cursor)]
    (if-let [row (lines/scan lines row direction non-empty-line?)]
      (let [line (lines row)
            nearest-col (binary-nearest-by :index line align-char-index)
            col (or (some #(when (non-empty-region? (line %)) %)
                          (range nearest-col (count line)))
                    (some #(when (non-empty-region? (line %)) %)
                          (range (dec nearest-col) 0 -1)))
            cursor [row col]]
        (-> layout
            (set-cursor cursor :anchor with-anchor :align false)
            ensure-cursor-visible))
      layout)))

(defn- cursor-to-start-of-selection [layout]
  (let [start (cursor/min (:cursor layout) (:anchor layout))]
    (-> layout
        (set-cursor start)
        ensure-cursor-visible)))

(defn- cursor-to-end-of-selection [layout]
  (let [end (cursor/max (:cursor layout) (:anchor layout))]
    (-> layout
        (set-cursor end)
        ensure-cursor-visible)))

(defn- cursor-to-end-of-line [layout with-anchor]
  (let [{:keys [lines cursor]} layout
        [row col] cursor
        line (lines row)]
    (if-let [new-col (some #(when (non-empty-region? (line %)) %)
                           (range (dec (count line)) (dec col) -1))]
      (let [cursor [row new-col]]
        (-> layout
            (set-cursor cursor :anchor with-anchor)
            ensure-cursor-visible))
      layout)))

(defn- cursor-to-beginning-of-line [layout with-anchor]
  (let [{:keys [lines cursor]} layout
        [row col] cursor
        line (lines row)]
    (if-let [new-col (some #(when (non-empty-region? (line %)) %) (range 0 (inc col)))]
      (let [cursor [row new-col]]
        (-> layout
            (set-cursor cursor :anchor with-anchor)
            ensure-cursor-visible))
      layout)))

(defn- string-builder
  ([] (StringBuilder.))
  ([^StringBuilder ret] (.toString ret))
  ([^StringBuilder acc in] (.append acc in)))

(defn- copy-selection! [layout]
  (fx/on-fx-thread
    (let [{:keys [cursor anchor lines]} layout
          from (cursor/min cursor anchor)
          to (cursor/max cursor anchor)
          text (transduce
                 (comp
                   (interpose ::newline)
                   (mapcat (fn [row]
                             (case row
                               ::newline [{:segments [{:text "\n"}]}]
                               (let [line (lines row)
                                     start-col (if (= row (cursor/row from))
                                                 (cursor/col from)
                                                 0)
                                     end-col (if (= row (cursor/row to))
                                               (cursor/col to)
                                               (dec (count line)))]
                                 (subvec line start-col (inc end-col))))))
                   (mapcat :segments)
                   (map :text))
                 string-builder
                 (range (cursor/row from) (inc (cursor/row to))))
          clipboard (Clipboard/getSystemClipboard)
          content (doto (ClipboardContent.)
                    (.putString text))]
      (.setContent clipboard content)))
  layout)

(defn- handle-key-pressed [layout ^KeyEvent event]
  (let [code (.getCode event)
        shortcut (.isShortcutDown event)
        with-anchor (not (.isShiftDown event))
        {:keys [cursor anchor]} layout]
    (condp = code
      KeyCode/ESCAPE
      (cond-> layout cursor (remove-cursor))

      KeyCode/UP
      (cond
        shortcut (update layout :scroll-y + (arrow-scroll layout :canvas-height))
        (not cursor) (introduce-cursor-at-bottom-of-screen layout)
        (and with-anchor (not= cursor anchor)) (cursor-to-start-of-selection layout)
        :else (move-cursor-vertically layout with-anchor dec))

      KeyCode/DOWN
      (cond
        shortcut (update layout :scroll-y - (arrow-scroll layout :canvas-height))
        (not cursor) (introduce-cursor-at-top-of-screen layout)
        (and with-anchor (not= cursor anchor)) (cursor-to-end-of-selection layout)
        :else (move-cursor-vertically layout with-anchor inc))

      KeyCode/LEFT
      (cond
        shortcut (update layout :scroll-x + (arrow-scroll layout :canvas-width))
        (not cursor) (introduce-cursor-at-bottom-of-screen layout)
        (and with-anchor (not= cursor anchor)) (cursor-to-start-of-selection layout)
        :else (move-cursor-horizontally layout with-anchor dec))

      KeyCode/RIGHT
      (cond
        shortcut (update layout :scroll-x - (arrow-scroll layout :canvas-width))
        (not cursor) (introduce-cursor-at-bottom-of-screen layout)
        (and with-anchor (not= cursor anchor)) (cursor-to-end-of-selection layout)
        :else (move-cursor-horizontally layout with-anchor inc))

      KeyCode/PAGE_UP
      (update layout :scroll-y + (page-scroll layout))

      KeyCode/PAGE_DOWN
      (update layout :scroll-y - (page-scroll layout))

      KeyCode/HOME
      (cond
        shortcut (-> layout
                     (assoc :scroll-y 0)
                     (cond-> cursor (dissoc :cursor :anchor :align-char-index)))
        (not cursor) (assoc layout :scroll-x 0)
        :else (cursor-to-beginning-of-line layout with-anchor))

      KeyCode/END
      (cond
        shortcut (-> layout
                     (assoc :scroll-y ##-Inf)
                     (cond-> cursor (dissoc :cursor :anchor :align-char-index)))
        (not cursor) (assoc layout :scroll-x ##-Inf)
        :else (cursor-to-end-of-line layout with-anchor))

      KeyCode/C
      (if (and (.isShortcutDown event) cursor)
        (copy-selection! layout)
        layout)

      KeyCode/A
      (if (.isShortcutDown event)
        (select-all layout)
        layout)

      KeyCode/SPACE
      (cond-> layout cursor (show-popup event))

      layout)))

(defmethod event/handle ::on-key-pressed [{:keys [^KeyEvent fx/event state path]}]
  (update-layout-fx state path handle-key-pressed event))

(defmethod event/handle ::hide-popup [{:keys [state path]}]
  (update-layout-fx state path dissoc :popup))

(defmethod event/handle ::on-popup-key-pressed [{:keys [^KeyEvent fx/event dispatch]}]
  (when (= KeyCode/ESCAPE (.getCode event))
    {:dispatch dispatch}))

(defn- consume-popup-event [^Event e]
  (if (instance? RedirectedEvent e)
    (.consume (.getOriginalEvent ^RedirectedEvent e))
    (.consume e)))

;; todo popup
;; - action selection (in scroll pane!) and filtering
;; - results are displayed in same popup (enter)
;; - results can be added to main output with closing popup (alt+enter)
;; - results can be cljfx markup
;; - what about changing popup size? height can be dynamic...

(defn- popup-view [{:keys [actions ^BoundingBox bounds path on-cancel font]}]
  (let [^Screen screen (first (Screen/getScreensForRectangle (.getMinX bounds)
                                                             (.getMinY bounds)
                                                             (.getWidth bounds)
                                                             (.getHeight bounds)))
        screen-bounds (.getVisualBounds screen)
        bounds-min-x (max (.getMinX screen-bounds) (.getMinX bounds))
        bounds-min-y (max (.getMinY screen-bounds) (.getMinY bounds))
        bounds (Rectangle2D. bounds-min-x
                             bounds-min-y
                             (- (min (.getMaxX screen-bounds) (.getMaxX bounds))
                                bounds-min-x)
                             (- (min (.getMaxY screen-bounds) (.getMaxY bounds))
                                bounds-min-y))
        content-width 300
        shadow-radius 10
        shadow-offset-y 5
        popup-width (+ content-width shadow-radius shadow-radius)
        space-below (- (.getMaxY screen-bounds)
                       (.getMaxY bounds))
        space-above (- (.getMinY bounds)
                       (.getMinY screen-bounds))
        popup-at-the-bottom (< space-above space-below)
        pref-anchor-x (-> (.getMinX bounds)
                          (+ (* (.getWidth bounds) 0.5))
                          (- (* popup-width 0.5)))
        visible-start-x (+ pref-anchor-x shadow-radius)
        visible-end-x (+ pref-anchor-x popup-width (- shadow-radius))
        anchor-fix-x (cond
                       (< visible-start-x (.getMinX screen-bounds))
                       (- (.getMinX screen-bounds) visible-start-x)

                       (> visible-end-x (.getMaxX screen-bounds))
                       (- (.getMaxX screen-bounds) visible-end-x)

                       :else
                       0)
        arrow-width 10
        arrow-height 10
        arrow-x (- (* content-width 0.5) anchor-fix-x)
        max-content-height (- (if popup-at-the-bottom
                                space-below
                                space-above)
                              arrow-height)]
    {:fx/type :popup
     :anchor-location (if popup-at-the-bottom :window-top-left :window-bottom-left)
     :anchor-x (+ pref-anchor-x anchor-fix-x)
     :anchor-y (if popup-at-the-bottom
                 (- (.getMaxY bounds) shadow-radius (- shadow-offset-y))
                 (+ (.getMinY bounds) shadow-radius shadow-offset-y))
     :auto-fix false
     :hide-on-escape false
     :event-handler consume-popup-event
     :content [{:fx/type :v-box
                :pref-width content-width
                :max-width content-width
                :focus-traversable true
                :on-key-pressed {::event/type ::on-popup-key-pressed :path path :dispatch on-cancel}
                :effect {:fx/type :drop-shadow
                         :radius shadow-radius
                         :offset-y shadow-offset-y
                         :color "#333"}
                :children (-> []
                              (cond-> popup-at-the-bottom
                                      (conj {:fx/type :polygon
                                             :v-box/margin {:left (- arrow-x (* arrow-width 0.5))}
                                             :fill "#999"
                                             :points [0 arrow-height
                                                      arrow-width arrow-height
                                                      (* arrow-width 0.5) 0]}))
                              (conj {:fx/type :v-box
                                     :max-height max-content-height
                                     :padding {:top 5 :bottom 5}
                                     :style {:-fx-background-color "#999"}
                                     :children (for [action actions]
                                                 {:fx/type :label
                                                  :font (font/jfx-font font :regular)
                                                  :wrap-text true
                                                  :text (:label action)})})
                              (cond-> (not popup-at-the-bottom)
                                      (conj {:fx/type :polygon
                                             :v-box/margin {:left (- arrow-x (* arrow-width 0.5))}
                                             :fill "#999"
                                             :points [0 0
                                                      arrow-width 0
                                                      (* arrow-width 0.5) arrow-height]})))}]}))

(defn view [{:keys [layout path]}]
  (let [{:keys [popup canvas-width canvas-height]} layout]
    (cond-> {:fx/type canvas-lifecycle
             :draw [draw layout]
             :width canvas-width
             :height canvas-height
             :focus-traversable true
             :on-key-pressed {::event/type ::on-key-pressed :path path}
             :on-mouse-dragged {::event/type ::on-mouse-dragged :path path}
             :on-mouse-pressed {::event/type ::on-mouse-pressed :path path}
             :on-mouse-released {::event/type ::on-mouse-released :path path}
             :on-width-changed {::event/type ::on-size-changed :key :canvas-width :path path}
             :on-height-changed {::event/type ::on-size-changed :key :canvas-height :path path}
             :on-scroll {::event/type ::on-scroll :path path}}

            popup
            (assoc :popup (assoc popup :fx/type popup-view
                                       :font (:font layout)
                                       :on-cancel {::event/type ::hide-popup :path path}
                                       :path (conj path :popup))))))

#_{:tag :ret
   :val {{:a 1
          :x []} {1 0 0 1}
         :b 2}
   :atom (atom 1)
   :e (RuntimeException. "beep")}

#_(with-meta {:a 1} {`clojure.core.protocols/nav vector})

#_(vec (map #(hash-map :n % % % :rand (rand)) (range 10000)))

#_(/ 1 0)

#_(.println System/out "asd\nbobob")

#_(font/make "Fantasque Sans Mono" 14.5)

#_(require 'clj-async-profiler.core)

#_(future
    (tap> (clj-async-profiler.core/profile-for 5)))