(ns reveal.font
  (:require [cljfx.coerce :as fx.coerce])
  (:import [com.sun.javafx.tk Toolkit]
           [com.sun.javafx.font PGFont FontResource FontStrike]
           [com.sun.javafx.scene.text FontHelper]
           [com.sun.javafx.geom.transform BaseTransform]))

(set! *warn-on-reflection* true)

(set! *unchecked-math* :warn-on-boxed)

(deftype Font [^javafx.scene.text.Font font ^double line-height ^double ascent char-width-cache])

(deftype Family [^Font regular ^Font bold ^Font italic ^Font bold-italic])

(def ^:private ^:const min-cached-char-width
  -42)

(def ^:private ^:const max-cached-char-width
  Double/MAX_VALUE)

(defn- make-char-width-cache [^FontStrike font-strike]
  (let [cache (double-array (inc (int Character/MAX_VALUE)) Double/MIN_VALUE)]
    (fn get-char-width [^Character character]
      (let [ch (unchecked-char character)
            i (unchecked-int ch)
            cached-width (aget cache i)]
        (if (= cached-width Double/MIN_VALUE)
          (let [width (.getCharAdvance font-strike ch)]
            (when (and (<= min-cached-char-width width)
                       (<= width max-cached-char-width))
              (aset cache i width))
            width)
          cached-width)))))

(defn- font ^Font [font]
  (let [^javafx.scene.text.Font font (fx.coerce/font font)
        metrics (.getFontMetrics (.getFontLoader (Toolkit/getToolkit)) font)
        strike (.getStrike ^PGFont (FontHelper/getNativeFont font)
                           BaseTransform/IDENTITY_TRANSFORM
                           FontResource/AA_GREYSCALE)]
    (Font. font
           (.getLineHeight metrics)
           (.getAscent metrics)
           (make-char-width-cache strike))))

(defn make [family-name size]
  (Family.
    (font {:family family-name :size size :posture :regular :weight :normal})
    (font {:family family-name :size size :posture :regular :weight :bold})
    (font {:family family-name :size size :posture :italic :weight :normal})
    (font {:family family-name :size size :posture :italic :weight :bold})))

(defn- get-font ^Font [^Family this kind]
  (case kind
    :italic (.-italic this)
    :bold (.-bold this)
    :bold-italic (.-bold_italic this)
    (.-regular this)))

(defn line-height ^double [^Family this]
  (.-line_height ^Font (.-regular this)))

(defn ascent ^double [this kind]
  (.-ascent ^Font (get-font this kind)))

(defn char-width ^double [this kind char]
  ((.-char_width_cache (get-font this kind)) char))

(defn jfx-font ^javafx.scene.text.Font [this kind]
  (.-font ^Font (get-font this kind)))
