package com.hbase.haxwell;

import com.google.common.base.Preconditions;
import com.hbase.haxwell.api.HaxwellEventDataExtractor;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.util.Bytes;

public class HaxwellDefaultEventDataExtrator implements HaxwellEventDataExtractor {

    private final byte[] tableName;
    private final byte[] columnFamily;
    private final byte[] columnQualifier;

    /**
     * Construct with the table and column information from which payload data should be extracted.
     *
     * @param tableName name of the table on which mutation events will include payload data
     * @param columnFamily column family that will include payload data
     * @param columnQualifier column qualifier of the cell from which the payload data will be extracted
     */
    public HaxwellDefaultEventDataExtrator(byte[] tableName, byte[] columnFamily, byte[] columnQualifier) {
        Preconditions.checkNotNull(tableName, "tableName cannot be null");
        Preconditions.checkNotNull(columnFamily, "columnFamily cannot be null");
        Preconditions.checkNotNull(columnQualifier, "columnQualifier cannot be null");

        this.tableName = tableName;
        this.columnFamily = columnFamily;
        this.columnQualifier = columnQualifier;
    }

    /**
     * Extract the payload data from a KeyValue.
     * <p>
     * Data will only be extracted if it matches the configured table, column family, and column qualifiers. If no
     * payload data can be extracted, null will be returned.
     *
     * @param tableName table to which the {@code KeyValue} is being applied
     * @param keyValue contains a (partial) row mutation which may include payload data
     * @return the extracted payload data, or null if no payload data is included in the supplied {@code KeyValue}
     */
    @Override
    public byte[] extract(byte[] tableName, KeyValue keyValue) {
        if (Bytes.equals(this.tableName, tableName) && CellUtil.matchingColumn(keyValue, columnFamily, columnQualifier)) {
            return CellUtil.cloneValue(keyValue);
        } else {
            return null;
        }

    }
}
