(ns contrib.ednish2
  (:require clojure.set
            clojure.string
            clojure.edn
            contrib.rfc3986
            [contrib.debug :as dbg]
            [hyperfiddle.rcf :refer [tests]]))

;; like contrib.ednish but escapes special chars
;; i.e. space vs comma, lists vs maps, exclam vs slash all work correctly
;; the cost? Uglier encoded string

; https://tools.ietf.org/html/rfc2396#section-2.4.3
;
; unwise = "{" | "}" | "|" | "\" | "^" | "[" | "]" | "`"

(def escape-char \!)
(def -edn-dialect-mappings
  {\space \,
   \# \~
   \' \’ ; \u2019 – typographic apostrophe (vs typewriter). URL safe and eye friendly. Prevents clashes with `"`.
   \" \'
   \/ \.
   ;; \{ \( \} \)
   \[ \(
   \] \)
   escape-char escape-char              ; self escape
   })
(def -edn-dialect-mappings-reverse (clojure.set/map-invert -edn-dialect-mappings))

;; Paste this into chrome and it will display properly
;; https://ko.wikipedia.org/wiki/%EC%9C%84%ED%82%A4%EB%B0%B1%EA%B3%BC:%EB%8C%80%EB%AC%B8

(defn ->str-builder []
  #?(:clj (let [sb (new StringBuilder)]
            (fn
              ([] (.toString sb))
              ([x] (.append sb x) nil)))
     :cljs (let [sb #js []]
             (fn
               ([] (.join sb ""))
               ([x] (.push sb x) nil)))))

(defn ?escape [c] (if-some [c2 (get -edn-dialect-mappings c)] (str escape-char c2) c))

(defn encode "Re-encode an edn-string to url-safe dialect of edn-ish. Vectors, sets and maps
coalesce into lists and are not disambiguated."
  [edn-str]
  (let [sb (->str-builder)]
    (run! (fn [c] (sb (?escape c))) edn-str)
    (sb)))

(defn decode [ednish-str]
  (let [sb (->str-builder)]
    (reduce (fn [escaped? c]
              (cond escaped?          (do (sb (get -edn-dialect-mappings-reverse c c)) false)
                    (= escape-char c) true
                    :else             (sb c)))
      false ednish-str)
    (sb)))

(tests
  (encode (pr-str :hyperfiddle.blog/post)) := ":hyperfiddle.blog!.post"
  (encode "pour l'amour") := "pour!,l!’amour"
  (encode "pour l’amour") := "pour!,l’amour" ; note the apostrophe
  (decode (encode "pour l'amour")) := "pour l'amour"
  (decode (encode "pour l’amour")) := "pour l’amour" ; \u2019
  (encode (pr-str :a!b)) := ":a!!b"
  (encode (pr-str :a/b)) := ":a!.b"     ; no clash
  (decode (encode (pr-str :a/b))) := ":a/b" ; decodes back
  (encode (pr-str "kobe")) := "!'kobe!'"
  (encode (pr-str #{"events" "news"})) := "!~{!'news!'!,!'events!'}"
  (encode (pr-str #uuid "07655f77-608d-472b-bc5e-86fcecc40b00"))
  := "!~uuid!,!'07655f77-608d-472b-bc5e-86fcecc40b00!'")

(defn encode-uri [x]
  (binding [*print-namespace-maps* true] ; unify platform default settings for more compact URLs. Defaults to: true in clj, false in cljs.
    (-> x pr-str encode contrib.rfc3986/encode-pchar)))

(def decode-uri (comp clojure.edn/read-string decode contrib.rfc3986/decode-pchar))

(tests
  "url encoding"
  (encode-uri "|") := "!'%7C!'"
  (decode-uri (encode-uri "|")) := "|"

  (encode-uri "!") := "!'!!!'"
  (decode-uri (encode-uri "/")) := "/"

  (encode-uri "$&'[]()*+,;=|") := "!'$&!’!(!)()*+,;=%7C!'"
  (decode-uri (encode-uri "$&'[]()*+,;=|")) := "$&'[]()*+,;=|"

  (encode-uri `(Toggle)) := "(contrib.ednish2!.Toggle)"
  (decode-uri "(contrib.ednish2!.Toggle)") := '(contrib.ednish2/Toggle)

  (encode-uri {:foo :bar}) := "%7B:foo!,:bar%7D"
  (encode-uri {::foo :bar, :baz :asdf}) := "%7B:contrib.ednish2!.foo!,:bar,!,:baz!,:asdf%7D"
  (encode-uri {::foo :bar, ::baz :asdf}) := "!~:contrib.ednish2%7B:foo!,:bar,!,:baz!,:asdf%7D"
  )

;(tests -- No reader function for tag uri -- this test passes in hf-2020
;  "ednish-tunneling"
;  (def v #uri "datomic:free://datomic:4334/~dustin.getz")
;  (def encoded (encode-ednish (pr-str v)))
;  (pr-str v) := _
;  (encode-ednish (pr-str v)) := "~uri,'datomic:free:!!datomic:4334!~dustin.getz'"
;  ;(is (= (decode-ednish encoded) (pr-str v)))
;  ; "#uri \"datomic:free://datomic:4334/#dustin.getz\""
;  )

(defn discard-leading-slash [path]
  (if (clojure.string/starts-with? path "/") (subs path 1) path))

(defn decode-path [path read-edn-str]
   {:pre [(string? path) (some? read-edn-str)]}
   (when-not (= path "/")
     (let [path (discard-leading-slash path)]
       (decode-uri path))))
