(ns clj-utils.user
  (:require [clojure.data.codec.base64 :as b64]
            [clojure.string :as str]))

; https://gist.github.com/pingles/1235344
(defn bytes-to-int
  ([bytes]
     (bytes-to-int bytes 0))
  ([bytes offset]
     (apply +
            (map (fn [i]
                   (let [shift (* (- 4 1 i)
                                  8)]
                     (bit-shift-left (bit-and (nth bytes (+ i offset))
                                              0x000000FF)
                                     shift)))
                 (range 0 4)))))

(defn user-id-convert
  "Convert user id from bytearray to web safe string"
  ^String [^com.google.protobuf.ByteString$LiteralByteString user-bytes]
  (-> user-bytes
      (.toByteArray)
      ^bytes (b64/encode)
      (String. "UTF-8")
      (str/replace "+" "-")
      (str/replace "/" "_")))

(defn device-id-convert
  "Convert b64 device ID to UUID"
  [device-id]
  device-id)

(defn extract-user-dc
  "Extract Data Centre ID from user_id.
  It is is encoded in the first 5 bits of the 5th byte of the user_id.
  Unknown=0, EU=1, US=2, AP=3."
  [^java.lang.String user_id]
  (bit-and 31 (nth (b64/decode (.getBytes user_id)) 4)))

(defn extract-user-ts
  "Extract user creation UNIX timetamp from user_id.
  It is encoded in the first 4 bytes of the user_id."
  [^java.lang.String user_id]
  (bytes-to-int (take 4 (b64/decode (.getBytes user_id)))))
