(ns crypto.core
  (:require
    [pallet.thread-expr :as th])
  (:import
    [java.security KeyStore PrivateKey PublicKey Security KeyPair KeyFactory]
    [java.security.spec X509EncodedKeySpec]
    [java.util Base64]
    [java.security.cert Certificate]
    [java.util Enumeration]
    [javax.crypto Cipher]
    [java.io ByteArrayOutputStream]
    [java.nio ByteBuffer]
    [sun.security.pkcs11 SunPKCS11]))

(defmulti get-key
  (fn [_ {:keys [keystore-type]}] keystore-type))

;only public rsa
;openssl x509 -pubkey -noout -in crt.crt  > pubkey.pem
;remove header, tail and put everything in *one* line
(defmethod get-key :file
  [_ {:keys [file-path]}]
  (let [x509spec (->> file-path
                      slurp
                      clojure.string/trim
                      (.decode (Base64/getDecoder))
                      (X509EncodedKeySpec.))]
    (-> (KeyFactory/getInstance "RSA")
        (.generatePublic x509spec))))

(defn- extract-key [ks ks-password key-type alias]
  (condp = key-type
    :public (-> ks (.getCertificate alias) (.getPublicKey))
    :private (.getKey ks alias (.toCharArray ks-password))
    :key-pair (KeyPair. (extract-key ks ks-password :public alias)
                        (extract-key ks ks-password :private alias))))

;pkcs11 keystore
(defmethod get-key :pkcs11
  [provider {:keys [keystore-password alias key-type]}]
  (let [keystore (KeyStore/getInstance "PKCS11" provider)]
    (do (.load keystore nil (.toCharArray keystore-password))
        (extract-key keystore keystore-password key-type alias))))

(defmethod get-key :pkcs12
  [provider {:keys [keystore-path keystore-password alias key-type]}]
  (with-open [fio (clojure.java.io/input-stream keystore-path)]
    (let [keystore (KeyStore/getInstance "PKCS12" provider)]
      (do (.load keystore fio (.toCharArray keystore-password))
          (extract-key keystore keystore-password key-type alias)))))

(defmethod get-key :jceks
  [provider {:keys [keystore-path keystore-password alias key-type]}]
  (with-open [fio (clojure.java.io/input-stream keystore-path)]
    (let [keystore (KeyStore/getInstance "JCEKS" provider)]
      (do (.load keystore fio (.toCharArray keystore-password))
          (extract-key keystore keystore-password key-type alias)))))

(defn crypt
  [{:keys [key-spec provider-name transformation base64-input? base64-output?]} cipher-mode data]
  (let [provider (Security/getProvider provider-name)
        key (get-key provider key-spec)
        cipher (Cipher/getInstance transformation provider)]
    (do (.init cipher cipher-mode key)
        (->> data
             (th/when->> base64-input?
                         (.decode (Base64/getDecoder)))
             (.doFinal cipher)
             (th/when->> base64-output?
                         (.encodeToString (Base64/getEncoder)))))))

(defn encrypt
  [params data]
  (crypt params Cipher/ENCRYPT_MODE data))

(defn decrypt
  [params data]
  (crypt params Cipher/DECRYPT_MODE data))

