package com.kahui.spiders.proxy;

import org.apache.log4j.Logger;

import java.util.*;
import java.util.concurrent.locks.ReentrantReadWriteLock;


public class ProxyPool {
    private static final Logger logger = Logger.getLogger(ProxyPool.class);
    public static final int SLEEP_INTERVAL_MS = 1000;
    public static final int MAX_TRY_COUNT = 5;
    public static final int MIN_USED_INTERVAL_MS = 10000;
    public static final double MAX_ERROR_RATE = 0.50;
    public static final int COOL_DOWN_MS = 30 * 60 * 1000;
    private final TreeSet<Proxy> tree = new TreeSet<Proxy>(new ProxyComparator());
    private final ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    private final ReentrantReadWriteLock.WriteLock writeLock = readWriteLock.writeLock();
    private final ReentrantReadWriteLock.ReadLock readLock = readWriteLock.readLock();
    private final int coolDownMs;

    public static class ProxyComparator implements Comparator<Proxy> {
        @Override
        public int compare(Proxy o1, Proxy o2) {
            int timeComp = compare(o1.getLastUseTime(), o2.getLastUseTime());
            if (timeComp != 0) {
                return timeComp;
            } else {
                return o1.getId().compareTo(o2.getId());
            }
        }

        public static int compare(long x, long y) {
            return (x < y) ? -1 : ((x == y) ? 0 : 1);
        }
    }

    public ProxyPool() {
        this(COOL_DOWN_MS);
    }

    public ProxyPool(int coolDownMs) {
        this.coolDownMs = coolDownMs;
        logger.info("Use coolDownMs:" + coolDownMs);
    }

    /**
     * @param proxy
     */
    public void put(Proxy proxy) {
        put(proxy, System.currentTimeMillis());
    }

    /**
     * @param proxy
     * @param lastUsedTime
     */
    void put(final Proxy proxy, final long lastUsedTime) {
        if (proxy == null) {
            throw new IllegalArgumentException("proxy is null");
        }
        long usedCount = proxy.getUsedCount().get();
        long errorCount = proxy.getErrCount().get();

        double errorRate = 0;
        if (usedCount >= 10) {
            errorRate = (double) errorCount / (usedCount);
            proxy.getUsedCount().set(0);
            proxy.getErrCount().set(0);
        }
        try {
            long usedTime = lastUsedTime;
            writeLock.lock();
            tree.remove(proxy);
            if (errorRate >= MAX_ERROR_RATE) {
                usedTime = System.currentTimeMillis() + this.coolDownMs;
                logger.warn("The proxy " + proxy + " error rate (" + errorRate + ") >=" + MAX_ERROR_RATE + ",will be cool down " + this.coolDownMs + "ms");
            }
            Proxy entry = new Proxy(proxy.getHost(), proxy.getPort(), usedTime, proxy.getUser(), proxy.getPassword(), proxy.getSessionId(), proxy.getErrCount(), proxy.getUsedCount());
            tree.add(entry);
        } finally {
            writeLock.unlock();
        }
    }

    public Proxy get() {
        Proxy proxy = null;
        int tryCount = 0;
        while (proxy == null) {
            tryCount++;
            int size = 0;
            try {
                writeLock.lock();
                size = tree.size();
                proxy = tree.pollFirst();
            } finally {
                writeLock.unlock();
            }
            if (proxy == null) {
                if (tryCount > MAX_TRY_COUNT) {
                    break;
                }
                logger.warn("Can't get the proxy ,sleep " + SLEEP_INTERVAL_MS + " with try count " + tryCount + " size" + size);
                try {
                    Thread.sleep(SLEEP_INTERVAL_MS);
                } catch (InterruptedException e) {
                    logger.error("Catch interruped exception,break it.", e);
                    Thread.currentThread().interrupt();
                    break;
                }
            } else {
                break;
            }
        }
        if (proxy == null) {
            throw new RuntimeException("Can't get proxy");
        }
        long waitTime = MIN_USED_INTERVAL_MS - (System.currentTimeMillis() - proxy.getLastUseTime());
        if (waitTime > 0) {
            try {
                Thread.sleep(waitTime);
            } catch (InterruptedException e) {
                logger.error("Catch interruped exception,break it.", e);
                Thread.currentThread().interrupt();
            }
        }
        proxy.getUsedCount().incrementAndGet();
        return proxy;
    }

    @Override
    public String toString() {
        try {
            readLock.lock();
            return "ProxyPool{" +
                    "tree=" + tree +
                    "size=" + tree.size() +
                    '}';
        } finally {
            readLock.unlock();
        }
    }
}
